
#import "./General.js"

function ATVASettingsAboutPage(title, isWiFi) {
    ATVAPage.call(this, title);
    if (typeof isWiFi == 'undefined') {
        this._wifi = false;
    }
    else {
        if (isWiFi) {
            this._wifi = true;
        }
        else {
            this._wifi = false;
        }
    }
    this._screenWidth = atvaTarget.resolutionWidth();
    this._screenHeight = atvaTarget.resolutionHeight();

    this._titleCenters = undefined;
    this._listCenters = undefined;
    this._copyrightCenters = undefined;

    this.kNameString = "Name";
    this.kModelString = "Model";
    this.kSerialNumberString = "Serial Number";
    this.kAppleTVSoftwareString = "Apple TV Software";
    this.kOSBuildVersionString = "OS Build Version";
    this.kTVResolutionString = "TV Resolution";
    this.kIPAddressString = "IP Address";
    this.kEthernetAddressString = "Ethernet Address";
    this.kBonjourDeviceIDString = "Bonjour Device ID";
    this.kWiFiNetworkString = "Wi-Fi Network";
    this.kWiFiAddressString = "Wi-Fi Address";
    this.kSignalStrengthString = "Signal Strength";
    this.kWiFiBSSIDString = "Wi-Fi BSSID";
    this.kWiFiChannelString = "Wi-Fi Channel";
    this.infoValues = {};

    this.kSelectState00 = 0;
    this.kSelectState01 = 1;
    this.kUpState00 = 0;
    this.kUpState01 = 1;
    this.kUpState02 = 2;
    this.kUpState03 = 3;
    this.pageState = {'selectState':0, 'upState':0}

    this.calculateCenters = function calculateCenters() {
        if ( ! this._titleCenters ) {
            this._titleCenters = ATVAPageList.titleCenters(this._screenWidth, this._screenHeight, "Right");
        }
        if ( ! this._listCenters ) {
            this._listCenters = ATVAPageList.columnCenters(this._screenWidth, this._screenHeight, "Right");
        }
        if ( ! this._copyrightCenters ) {
            this._copyrightCenters = ATVAPageList.columnCenters(this._screenWidth, this._screenHeight, "Center");
        }
    }
    // default the calulations
    this.calculateCenters();

    this.isInfoElement = function isInfoElement(element) {
        this.calculateCenters();

        // The element must be UIAElement
        if (element.attributes().class != "UIAElement") {
            return(false);
        }

        if ( ! element.name() ) {
            // Info element cannot be empty
            return(false);
        }

        var theRect = element.rect();

        // If the element is zero size, it cannot be seen and is not part of the menu.
        if ( (theRect.size.width == 0) || (theRect.size.height == 0) ) {
            return(false);
        }
        // If the element is width is the size of the screen this is most likely a over lay.
        // Ex. There is a UIAElement surrounding the staticText at the bottom of the screen.
        if ( theRect.size.width == this._screenWidth ) {
            return(false);
        }

        // The elements x center most be in the list
        if (Math.abs(this._listCenters.x - (theRect.origin.x + (theRect.size.width / 2))) > this._listCenters.xlowrange) {
            if (Math.abs(this._listCenters.x - (theRect.origin.x + (theRect.size.width / 2))) > this._listCenters.xhighrange) {
                return(false);
            }
        }

        // This is the element
        return(true);
    }

    this.isCopyrights = function isCopyrights(element) {
        this.calculateCenters();

        // The element must be UIAStaticText
        if (element.attributes().class != "UIAStaticText") {
            return(false);
        }

        var theRect = element.rect();

        // If the element is zero size, it cannot be seen and is not part of the menu.
        if ( (theRect.size.width == 0) || (theRect.size.height == 0) ) {
            return(false);
        }
        // If the element is width is the size of the screen this is most likely a over lay.
        // Ex. There is a UIAElement surrounding the staticText at the bottom of the screen.
        if ( theRect.size.width == this._screenWidth ) {
            return(false);
        }

        // The elements x center most be in the list
        if (Math.abs(this._copyrightCenters.x - (theRect.origin.x + (theRect.size.width / 2))) > this._copyrightCenters.xlowrange) {
            if (Math.abs(this._copyrightCenters.x - (theRect.origin.x + (theRect.size.width / 2))) > this._copyrightCenters.xhighrange) {
                return(false);
            }
        }

        // This is the element
        return(true);
    }

    this.isTitle = function isTitle(element) {
        this.calculateCenters();

        // The element must be UIAStaticText
        if (element.attributes().class != "UIAStaticText") {
            return(false);
        }

        var theRect = element.rect();

        // If the element is zero size, it cannot be seen and is not part of the menu.
        if ( (theRect.size.width == 0) || (theRect.size.height == 0) ) {
            return(false);
        }

        // The elements x and y centers most be in the title area
        if (Math.abs(this._titleCenters.x - (theRect.origin.x + (theRect.size.width / 2))) > this._titleCenters.xlowrange) {
            if (Math.abs(this._listCenters.x - (theRect.origin.x + (theRect.size.width / 2))) > this._titleCenters.xhighrange) {
                return(false);
            }
        }
        if (Math.abs(this._titleCenters.y - (theRect.origin.y + (theRect.size.height / 2))) > this._titleCenters.ylowrange) {
            if (Math.abs(this._titleCenters.y - (theRect.origin.y + (theRect.size.height / 2))) > this._titleCenters.yhighrange) {
                return(false);
            }
        }

        // This is the title
        return(true);
    }

    this.menuTitle = function menuTitle() {
        var theElements = this.staticTexts().toArray();
        for (var i = 0; i < theElements.length; ++i) {
            if (this.isTitle(theElements[i])) {
                return(theElements[i]);
            }
        }
        return(undefined);
    }

    this.copyrights = function copyrights() {
        var theElements = this.staticTexts().toArray();
        for (var i = 0; i < theElements.length; ++i) {
            if (this.isCopyrights(theElements[i])) {
                return(theElements[i]);
            }
        }
        return(undefined);
    }

    this.infoElement = function infoElement() {
        var theElements = this.elements().toArray();
        for (var i = 0; i < theElements.length; ++i) {
            if (this.isInfoElement(theElements[i])) {
                return(theElements[i]);
            }
        }
        return(undefined);
    }

    this.isPageDisplayed = function isPageDisplayed() {
        var titleElement = this.menuTitle();
        if (titleElement && (titleElement.name() == this._title)) {
            return(true);
        }
        else {
            return(false);
        }
    }

    this.currentInfo = function currentInfo() {
        var infoElement = this.infoElement();
        var tmpArray = infoElement.name().split(/,\s*/);
        var info = {};

        var simpleValues = [this.kNameString, this.kModelString, this.kAppleTVSoftwareString, this.kOSBuildVersionString, this.kTVResolutionString, this.kIPAddressString, this.kWiFiNetworkString, this.kSignalStrengthString, this.kWiFiChannelString];
        var addressValues = [this.kEthernetAddressString, this.kBonjourDeviceIDString, this.kWiFiAddressString, this.kWiFiBSSIDString];
        var index;

        for (var i = 0; i < tmpArray.length; ++i) {
            if ( (index = simpleValues.indexOf(tmpArray[i])) > -1 ) {
                if ( (i + 1) < tmpArray.length ) {
                    info[tmpArray[i]] = tmpArray[i+1];
                    ++i;
                }
                else {
                    UIALogger.logError("ATVASettingsAboutPage.currentInfo(): no data for " + tmpArray[i]);
                }
            }
            else if ( (index = addressValues.indexOf(tmpArray[i])) > -1 ) {
                // Most Addresses will always be the next 6 items in the array, but some WiFi
                // addresses will be just one item if WiFi is not configured.
                // Wi-Fi BSSID, 0, 1B, 63, 2B, A4, A5
                // Wi-Fi BSSID, N/A

                if ( ((i + 1) < tmpArray.length) && (tmpArray[i+1] == 'N/A') ) {
                    info[tmpArray[i]] = tmpArray[i+1];
                    ++i;
                }
                else if ( (i + 6) < tmpArray.length ) {
                    info[tmpArray[i]] = tmpArray.slice(i+1, i+7).join(':');
                    i+=6;
                }
                else {
                    UIALogger.logError("ATVASettingsAboutPage.currentInfo(): not enough data for " + tmpArray[i]);
                }
            }
            else if (tmpArray[i] == this.kSerialNumberString) {
                if ( (i + 1) < tmpArray.length ) {
                    // get rid of spaces
                    info[tmpArray[i]] = tmpArray[i+1].split(/\s+/).join('');
                    ++i;
                }
                else {
                    UIALogger.logError("ATVASettingsAboutPage.currentInfo(): no data for " + tmpArray[i]);
                }
            }
            else {
                UIALogger.logError("ATVASettingsAboutPage.currentInfo(): unknown data " + tmpArray[i]);
            }
        }

        return(info);
    }

    this.mergeInfo = function mergeInfo(info) {
        var keys = Object.keys(info);

        for (var i = 0; i < keys.length; ++i) {
            if (keys[i] in this.infoValues) {
                if (this.infoValues[keys[i]] != info[keys[i]]) {
                    UIALogger.logError("ATVASettingsAboutPage.mergeInfo(): value changed for " + keys[i] + " from " + this.infoValues[keys[i]] + " to " + info[keys[i]]);
                    this.infoValues[keys[i]] = info[keys[i]];
                }
            }
            else {
                this.infoValues[keys[i]] = info[keys[i]];
            }
        }
    }

    this.stateFromInfo = function stateFromInfo(info) {
        var state = {'selectState':undefined, 'upState':undefined};

        if (this.kAppleTVSoftwareString in info) {
            state['selectState'] = this.kSelectState00;
        }
        else if (this.kOSBuildVersionString in info) {
            state['selectState'] = this.kSelectState01;
        }
        else {
            UIALogger.logError("ATVASettingsAboutPage.stateFromInfo(): cannot determine select state");
        }

        if (this.kWiFiBSSIDString in info) {
            state['upState'] = this.kUpState01;
        }
        else if (this.kWiFiChannelString in info) {
            state['upState'] = this.kUpState02;
        }
        else if (this.kBonjourDeviceIDString in info) {
            state['upState'] = this.kUpState03;
        }
        else {
            state['upState'] = this.kUpState00;
        }

        return(state);
    }

    this.setCurrentPageState = function setCurrentPageState() {
        var info = this.currentInfo();
        var state = this.stateFromInfo(info);
        this.mergeInfo(info);
        this.pageState['selectState'] = state['selectState'];
        this.pageState['upState'] = state['upState'];
    }

    this.nextState = function nextState(buttonPressed) {
        if ( buttonPressed == "select" ) {
            if (this.pageState['selectState'] == this.kSelectState00) {
                this.pageState['selectState'] = this.kSelectState01;
            }
            else {
                this.pageState['selectState'] = this.kSelectState00;
            }
        }
        else if ( buttonPressed == "up" ) {
            if (this.pageState['upState'] == this.kUpState00) {
                if (this._wifi) {
                    this.pageState['upState'] = this.kUpState01;
                }
                else {
                    this.pageState['upState'] = this.kUpState03;
                }
            }
            else if (this.pageState['upState'] == this.kUpState01) {
                this.pageState['upState'] = this.kUpState02;
            }
            else if (this.pageState['upState'] == this.kUpState02) {
                this.pageState['upState'] = this.kUpState03;
            }
            else if (this.pageState['upState'] == this.kUpState02) {
                this.pageState['upState'] = this.kUpState00;
            }
            else {
                this.pageState['upState'] = this.kUpState00;
            }
        }
    }

    this.compareStateWithState = function compareStateWithState(state) {
        var stateAreEqual = true;
        if (this.pageState['selectState'] != state['selectState']) {
            UIALogger.logError("ATVASettingsAboutPage.compareStateWithState(): select states do not match " + this.pageState['selectState'] + " != " + state['selectState']);
            stateAreEqual = false;
        }
        if (this.pageState['upState'] != state['upState']) {
            UIALogger.logError("ATVASettingsAboutPage.compareStateWithState(): up states do not match " + this.pageState['upState'] + " != " + state['upState']);
            stateAreEqual = false;
        }
        return(stateAreEqual);
    }

    // Side effect of verifying states is that all info is loaded.
    this.verifyStates = function verifyStates() {
        this.setCurrentPageState();
        var info;
        var state;
        var testPassed = true;

        // select states
        for (var i = 0; i < 3; ++i) {
            this.buttonSelect();
            this.nextState("select");

            info = this.currentInfo();
            this.mergeInfo(info);
            state = this.stateFromInfo(info);
            if ( ! this.compareStateWithState(state)) {
                this.printInfo(info);
                testPassed = false;
            }
        }

        var upCount = 3;
        if (this._wifi) {
            upCount = 5;
        }
        for (var i = 0; i < upCount; ++i) {
            this.buttonUp();
            this.nextState("up");

            info = this.currentInfo();
            this.mergeInfo(info);
            state = this.stateFromInfo(info);
            if ( ! this.compareStateWithState(state)) {
                this.printInfo(info);
                testPassed = false;
            }
        }

        return(testPassed);
    }

    this.printInfo = function printInfo(info) {
        if (typeof info == 'undefined') {
            info = this.infoValues;
        }
        var keys = [this.kNameString, this.kModelString, this.kSerialNumberString, this.kAppleTVSoftwareString, this.kOSBuildVersionString, this.kTVResolutionString, this.kIPAddressString, this.kEthernetAddressString, this.kBonjourDeviceIDString, this.kWiFiNetworkString, this.kWiFiAddressString, this.kSignalStrengthString, this.kWiFiBSSIDString, this.kWiFiChannelString];

        UIALogger.logMessage("About");
        for (var i = 0; i < keys.length; ++i) {
            if (keys[i] in info) {
                UIALogger.logMessage("    " + keys[i] + " = " + info[keys[i]]);
            }
        }
    }

    return(this);
}

// --------------------------------------------------------------------------------
//      Object: about
//              about is an ATVAApp object for interacting with the
//              Settings/General/About appliance on Apple TV.
// --------------------------------------------------------------------------------

if (typeof about == 'undefined') {
    about = new ATVAApp("About");
}

if (typeof about != 'undefined') {
    about._objectName = "about";

    // overriding appMenu(), and goTo() methods from base ATVAApp()
    about.appMenu = function appMenu(isWifi) {
        if ( ! this._appMenu ) {
            this._appMenu = new ATVASettingsAboutPage(this._title, isWifi);
        }
        return( this._appMenu );
    }

    about.goTo = function goTo(isWifi) {
        // Indexes are not saved when we go back to the Main Menu
        if (general.goToAndSelect(this._title)) {
            return(this.fromMenu(true));
        }
        UIALogger.logError(this._objectName + ".goTo(): failed to get to " + this._title);
        return(undefined);
    }
}
