
/**
 *       Function: vimeoTest_browseFeaturedChannelsAtAndPlayFor.
 *           Go to Vimeo>Featured Channels, go to row, column and play for for time
 *   
 *  \param row (string) - row of item to play
 *  \param column (string) - column of item to play
 *  \param secondsToPlay (string) - time to play item
 *   
 *  \return 1 if found, otherwise 0
 **/

function vimeoTest_browseFeaturedChannelsAtAndPlayFor() {
    if (this.argv.length < 3) {
        this.testError = "Row, column, and seconds to play are required";
        return this.testResult = 0;
    }
    var row = parseInt(this.argv[0]);
    var column = parseInt(this.argv[1]);
    var secondsToPlay = parseInt(this.argv[2]);

    var dp = vimeo.goToFeaturedRowColumn(row, column);
    if ( ! dp) {
        this.testError = "Failed to load movies detail page";
        return this.testResult = 0;
    }

    if ( ! dp.buttons()[dp.playItemName()].isValid()) {
        UIALogger.logMessage("This is not a movie");
        return this.testResult = 1;
    }

    if (secondsToPlay == -1) {
        if ( ! dp.playAndWaitForEnd(false)) {
            this.testError = "Failed to play movie " + dp.title();
            return this.testResult = 0;
        }
    }
    else if (secondsToPlay == -2) {
        // Browse only, do not play
    }
    else {
        if ( ! dp.play(false)) {
            this.testError = "Failed to play movie " + dp.title();
            return this.testResult = 0;
        }
        else {
            atvaTarget.delay(secondsToPlay);
            dp.dismissScreenSaver();
            dp.menuOutOfPage();
        }
    }
    return this.testResult = 1;
}


/**
 *       Function: vimeoTest_searchForVideo
 *           Search for video through Vimeo>Search.
 *   
 *  \param videoTitle (string) - video title
 *  \param videoAuthor (string) - video author
 *   
 *  \return 1 if found, otherwise 0
 **/

function vimeoTest_searchForVideo() {
    if (this.argv.length < 2) {
        this.testError = "No video title or author supplied.";
        return this.testResult = 0;
    }
    var videoTitle = this.argv[0];
    var videoAuthor = this.argv[1];
    if ( ! vimeo.searchForVideo(videoTitle, videoAuthor)) {
        this.testError = "Fail to search for movie";
        return this.testResult = 0;
    }
    else {
        return this.testResult = 1;
    }
}

/**
 *       Function: vimeoTest_searchForVideoAndPlay
 *           Search for video through Vimeo>Search, play movie from beginning and wait for it to complete.
 *   
 *  \param videoTitle (string) - video title
 *  \param videoAuthor (string) - video author
 *   
 *  \return 1 if movie played, otherwise 0
 **/

function vimeoTest_searchForVideoAndPlay() {
    if (this.argv.length < 2) {
        this.testError = "No video title or author supplied.";
        return this.testResult = 0;
    }
    var videoTitle = this.argv[0];
    var videoAuthor = this.argv[1];
    if ( ! vimeo.searchAndPlayVideoWaitForEnd(videoTitle, videoAuthor, false)) {
        this.testError = "Fail to search for and play video";
        return this.testResult = 0;
    }
    else {
        return this.testResult = 1;
    }
}

/**
 *       Function: vimeoTest_searchForVideoAndPlayFor
 *           Search for video through Vimeo>Search, play movie from beginning for the specified number of seconds.
 *           Notice that the time to play starts after the movie begins playback and does not include caching.
 *   
 *  \param videoTitle (string) - video title
 *  \param videoAuthor (string) - video author
 *  \param secondsToPlay (number) - seconds to play movie.
 *   
 *  \return 1 if movie played, otherwise 0
 **/

function vimeoTest_searchForVideoAndPlayFor() {
    if (this.argv.length < 3) {
        this.testError = "No video title, author, and time to play supplied";
        return this.testResult = 0;
    }
    var videoTitle = this.argv[0];
    var videoAuthor = this.argv[1];
    var secondsToPlay = this.argv[2];

    var dp = vimeo.searchAndPlayVideo(videoTitle, videoAuthor, false);

    if ( ! dp) {
        this.testError = "Fail to search for and play video";
        return this.testResult = 0;
    }
    else {
        atvaTarget.delay(secondsToPlay);
        dp.dismissScreenSaver();
        dp.menuOutOfPage();
        return this.testResult = 1;
    }
}

/**
 *       Function: vimeoTest_searchForVideoAndPlayAndPauseFor
 *           Search for video through Vimeo>Search, play movie from beginning. After playback starts, pause for
 *           secondsToPause. Notice that the time to pause starts after the movie begins playback and does not
 *           include caching.
 *   
 *  \param videoTitle (string) - video title
 *  \param videoAuthor (string) - video author
 *  \param secondsToPause (number) - seconds to play movie.
 *   
 *  \return 1 if movie played, otherwise 0
 **/

function vimeoTest_searchForVideoAndPlayAndPauseFor() {
    if (this.argv.length < 3) {
        this.testError = "No video title, author, and time to pause supplied";
        return this.testResult = 0;
    }
    var videoTitle = this.argv[0];
    var videoAuthor = this.argv[1];
    var secondsToPause = this.argv[2];

    var dp = vimeo.searchAndPlayVideo(videoTitle, videoAuthor, false);

    if ( ! dp) {
        this.testError = "Fail to search for and play video";
        return this.testResult = 0;
    }
    else {
        dp.buttonSelect();
        atvaTarget.delay(secondsToPause);
        dp.dismissScreenSaver();
        dp.menuOutOfPage();
        return this.testResult = 1;
    }
}
