/*
	File:		gpus_program.c

	Copyright:	(c) 1999-2010 by Apple Inc., all rights reserved.

	Writers:
		John Stauffer
		Bob Beretta
*/

#if !GLD_PROGRAM_ONLY
#error
#endif

#include "glr_config.h"

#include <OpenGLES/gl.h>
#include <OpenGLES/glext.h>
#include <OpenGLES/glOESAdditions.h>
#include <OpenGLES/glext_private.h>

#include <OpenGLES/gldState.h>
#include <OpenGLES/gldProgram.h>

#include <GPUSupport/gpus_internal.h>
#include <GPUSupport/gpus_io_data.h>
#include <GPUSupport/gpus_device.h>
#include <GPUSupport/gpus_shared.h>
#include <GPUSupport/gpus_context.h>
#include <GPUSupport/gpus_program.h>
#include <GPUSupport/gpus_vendor.h>
#include <GPUSupport/gpus_export.h>
#include <GPUSupport/gpus_cookies.h>
#include <GPUSupport/gpus_assert.h>
#include <GPUSupport/gpus_memory.h>
#include <GPUSupport/gpus_init.h>

#include <GPUSupport/GIO/GIOCTAssert.h>

#include <string.h>

#define ALWAYS_INLINE static inline __attribute__ ((always_inline))

void gpumSetCookieNoProgramExtern(GLRCookie *cookie)
{
	gpumSetCookieNoProgram(cookie);
}

ALWAYS_INLINE void glrSetCookieWithBits(GLDShareGroup shared,
	GLRCookie *cookie, uint32_t dirty_bits)
{
	#if GLR_PIPEPROG_ALL_DIRTY_MASK & GLD_PROGRAM_DATA_BIT
	if(dirty_bits & GLD_PROGRAM_DATA_BIT)
		cookie->data_cookie = ++shared->shr_data_cookie;
	#endif
	#if GLR_PIPEPROG_ALL_DIRTY_MASK & GLD_PROGRAM_LOCALS_BIT
	if(dirty_bits & GLD_PROGRAM_LOCALS_BIT)
		cookie->locals_cookie = ++shared->shr_locals_cookie;
	#endif
	#if GLR_PIPEPROG_ALL_DIRTY_MASK & GLD_PROGRAM_SAMPLERS_BIT
	if(dirty_bits & GLD_PROGRAM_SAMPLERS_BIT)
		cookie->samplers_cookie = ++shared->shr_samplers_cookie;
	#endif
	#if GLR_PIPEPROG_ALL_DIRTY_MASK & GLD_PROGRAM_UNIFORMBUFFER_BIT
	if(dirty_bits & GLD_PROGRAM_UNIFORMBUFFER_BIT)
		cookie->uniformbuffers_cookie = ++shared->shr_uniformbuffers_cookie;
	#endif
	#if GLR_PIPEPROG_ALL_CTX_DIRTY_MASK & GLD_PROGRAM_TEXTURES_BIT
	gpusAssert(!(dirty_bits & GLD_PROGRAM_TEXTURES_BIT));
	#endif
	#if GLR_PIPEPROG_ALL_CTX_DIRTY_MASK & GLD_PROGRAM_STATE_BIT
	gpusAssert(!(dirty_bits & GLD_PROGRAM_STATE_BIT));
	#endif
}

ALWAYS_INLINE uint32_t glrGetCookieDiffBits(GLRCookie *cookie1, GLRCookie *cookie2)
{
	uint32_t dirty_bits = 0U;
	#if GLR_PIPEPROG_ALL_DIRTY_MASK & GLD_PROGRAM_DATA_BIT
	if(cookie1->data_cookie != cookie2->data_cookie)
		dirty_bits |= GLD_PROGRAM_DATA_BIT;
	#endif
	#if GLR_PIPEPROG_ALL_DIRTY_MASK & GLD_PROGRAM_LOCALS_BIT
	if(cookie1->locals_cookie != cookie2->locals_cookie)
		dirty_bits |= GLD_PROGRAM_LOCALS_BIT;
	#endif
	#if GLR_PIPEPROG_ALL_DIRTY_MASK & GLD_PROGRAM_SAMPLERS_BIT
	if(cookie1->samplers_cookie != cookie2->samplers_cookie)
		dirty_bits |= GLD_PROGRAM_SAMPLERS_BIT;
	#endif
	#if GLR_PIPEPROG_ALL_DIRTY_MASK & GLD_PROGRAM_UNIFORMBUFFER_BIT
	if(cookie1->uniformbuffers_cookie != cookie2->uniformbuffers_cookie)
		dirty_bits |= GLD_PROGRAM_UNIFORMBUFFER_BIT;
	#endif
	#if GLR_PIPEPROG_ALL_CTX_DIRTY_MASK & GLD_PROGRAM_TEXTURES_BIT
	if(cookie1->textures_cookie != cookie2->textures_cookie)
		dirty_bits |= GLD_PROGRAM_TEXTURES_BIT;
	#endif
	#if GLR_PIPEPROG_ALL_CTX_DIRTY_MASK & GLD_PROGRAM_STATE_BIT
	if(cookie1->state_cookie != cookie2->state_cookie)
		dirty_bits |= GLD_PROGRAM_STATE_BIT;
	#endif
	#if GLR_PIPEPROG_ALL_CTX_DIRTY_MASK & GLD_PROGRAM_ALL_LOCALS_BIT
	if(cookie1->all_locals_cookie != cookie2->all_locals_cookie)
		dirty_bits |= GLD_PROGRAM_ALL_LOCALS_BIT;
	#endif
	return dirty_bits;
}

#if !GLR_HAS_VERTEX_SHADER
#define glrBindSysVertexProgram      NULL
#define glrUpdateShrSysVertexProgram NULL
#define glrTestCtxSysVertexProgram   NULL
#define glrUpdateCtxSysVertexProgram NULL

#define GLR_PIPEPROG_VERT_CTX_DIRTY_MASK 0
#define GLR_PIPEPROG_VERT_SHR_DIRTY_MASK 0
#endif

#if !GLR_HAS_TESSELLATION_SHADER
#define glrBindSysTessControlProgram         NULL
#define glrUpdateShrSysTessControlProgram    NULL
#define glrTestCtxSysTessControlProgram      NULL
#define glrUpdateCtxSysTessControlProgram    NULL
#define glrBindSysTessEvaluationProgram      NULL
#define glrUpdateShrSysTessEvaluationProgram NULL
#define glrTestCtxSysTessEvaluationProgram   NULL
#define glrUpdateCtxSysTessEvaluationProgram NULL

#define GLR_PIPEPROG_CTRL_CTX_DIRTY_MASK 0
#define GLR_PIPEPROG_CTRL_SHR_DIRTY_MASK 0
#define GLR_PIPEPROG_EVAL_CTX_DIRTY_MASK 0
#define GLR_PIPEPROG_EVAL_SHR_DIRTY_MASK 0
#endif

#if !GLR_HAS_GEOMETRY_SHADER
#define glrBindSysGeometryProgram      NULL
#define glrUpdateShrSysGeometryProgram NULL
#define glrTestCtxSysGeometryProgram   NULL
#define glrUpdateCtxSysGeometryProgram NULL

#define GLR_PIPEPROG_GEOM_CTX_DIRTY_MASK 0
#define GLR_PIPEPROG_GEOM_SHR_DIRTY_MASK 0
#endif

static void
(* const glrBindSysProgram[])(GLDContext ctx, GLDProgram program,
	GLDPipelineProgram pipeprog, GLRVendBoundPipeProg vendpipeprog,
	bitfield32_t *param_dirty, bitfield32_t dirty_bits) =
{
	glrBindSysVertexProgram,
	glrBindSysTessControlProgram,
	glrBindSysTessEvaluationProgram,
	glrBindSysGeometryProgram,
	glrBindSysFragmentProgram,
};
ct_assert(sizeof(glrBindSysProgram) / sizeof(glrBindSysProgram[0]) == GLD_NUM_PROGRAM_TARGETS);

#if GLR_PIPEPROG_HAS_SHR_BUILD
static GLRVendShrPipeProg
(* const glrUpdateShrSysProgram[])(GLDShareGroup shared,
	GLDProgram program,
	GLDPipelineProgram pipeprog,
	GLRVendShrPipeProg vendpipeprog,
	bitfield32_t dirty_bits) =
{
	glrUpdateShrSysVertexProgram,
	glrUpdateShrSysTessControlProgram,
	glrUpdateShrSysTessEvaluationProgram,
	glrUpdateShrSysGeometryProgram,
	glrUpdateShrSysFragmentProgram,
};
ct_assert(sizeof(glrUpdateShrSysProgram) / sizeof(glrUpdateShrSysProgram[0]) == GLD_NUM_PROGRAM_TARGETS);
#endif // GLR_PIPEPROG_HAS_SHR_BUILD

#if GLR_PIPEPROG_HAS_CTX_BUILD
#if GLR_PIPEPROG_CTX_TEST_STATE_CHANGE
static bool32_t
(* const glrTestCtxSysProgram[])(GLDContext ctx,
	GLDProgram program,
	GLDPipelineProgram pipeprog,
	#if GLR_PIPEPROG_HAS_SHR_BUILD
	GLRVendShrPipeProg sharedvendpp,
	#endif
	GLRVendCtxPipeProg vendpipeprog) =
{
	glrTestCtxSysVertexProgram,
	glrTestCtxSysTessControlProgram,
	glrTestCtxSysTessEvaluationProgram,
	glrTestCtxSysGeometryProgram,
	glrTestCtxSysFragmentProgram,
};
ct_assert(sizeof(glrTestCtxSysProgram) / sizeof(glrTestCtxSysProgram[0]) == GLD_NUM_PROGRAM_TARGETS);
#endif // GLR_PIPEPROG_CTX_TEST_STATE_CHANGE

static GLRVendCtxPipeProg
(* const glrUpdateCtxSysProgram[])(GLDContext ctx, GLDProgram program,
	GLDPipelineProgram pipeprog,
#if GLR_PIPEPROG_HAS_SHR_BUILD
	GLRVendShrPipeProg sharedvendpp,
#endif
	GLRVendCtxPipeProg vendpipeprog, bitfield32_t dirty_bits) =
{
	glrUpdateCtxSysVertexProgram,
	glrUpdateCtxSysTessControlProgram,
	glrUpdateCtxSysTessEvaluationProgram,
	glrUpdateCtxSysGeometryProgram,
	glrUpdateCtxSysFragmentProgram,
};
ct_assert(sizeof(glrUpdateCtxSysProgram) / sizeof(glrUpdateCtxSysProgram[0]) == GLD_NUM_PROGRAM_TARGETS);
#endif // GLR_PIPEPROG_HAS_CTX_BUILD

static const bitfield32_t ctx_dirty_masks[] = {
	GLR_PIPEPROG_VERT_CTX_DIRTY_MASK,
	GLR_PIPEPROG_CTRL_CTX_DIRTY_MASK,
	GLR_PIPEPROG_EVAL_CTX_DIRTY_MASK,
	GLR_PIPEPROG_GEOM_CTX_DIRTY_MASK,
	GLR_PIPEPROG_FRAG_CTX_DIRTY_MASK,
};
ct_assert(sizeof(ctx_dirty_masks) / sizeof(ctx_dirty_masks[0]) == GLD_NUM_PROGRAM_TARGETS);

static const bitfield32_t shr_dirty_masks[] = {
	GLR_PIPEPROG_VERT_SHR_DIRTY_MASK,
	GLR_PIPEPROG_CTRL_SHR_DIRTY_MASK,
	GLR_PIPEPROG_EVAL_SHR_DIRTY_MASK,
	GLR_PIPEPROG_GEOM_SHR_DIRTY_MASK,
	GLR_PIPEPROG_FRAG_SHR_DIRTY_MASK,
};
ct_assert(sizeof(shr_dirty_masks) / sizeof(shr_dirty_masks[0]) == GLD_NUM_PROGRAM_TARGETS);

static const bitfield32_t bind_dirty_masks[] = {
	GLR_PIPEPROG_VERT_BIND_DIRTY_MASK,
	GLR_PIPEPROG_CTRL_BIND_DIRTY_MASK,
	GLR_PIPEPROG_EVAL_BIND_DIRTY_MASK,
	GLR_PIPEPROG_GEOM_BIND_DIRTY_MASK,
	GLR_PIPEPROG_FRAG_BIND_DIRTY_MASK,
};
ct_assert(sizeof(bind_dirty_masks) / sizeof(bind_dirty_masks[0]) == GLD_NUM_PROGRAM_TARGETS);

// TODO verify behavior is the same for unsupported cases
static const bitfield32_t fallback_masks[] = {
	GPUS_FALLBACK_VERTEX_MASK,
	GPUS_FALLBACK_VERTEX_MASK,
	GPUS_FALLBACK_VERTEX_MASK,
	GPUS_FALLBACK_VERTEX_MASK,
	GPUS_FALLBACK_FRAGMENT_MASK,
};
ct_assert(sizeof(fallback_masks) / sizeof(fallback_masks[0]) == GLD_NUM_PROGRAM_TARGETS);

static const bitfield32_t fallback_bits[] = {
	GPUS_FALLBACK_VERTEX_FOR_VERT_PROG,
	GPUS_FALLBACK_VERTEX_FOR_CTRL_PROG,
	GPUS_FALLBACK_VERTEX_FOR_EVAL_PROG,
	GPUS_FALLBACK_VERTEX_FOR_GEOM_PROG,
	GPUS_FALLBACK_FRAGMENT_FOR_FRAG_PROG,
};
ct_assert(sizeof(fallback_bits) / sizeof(fallback_bits[0]) == GLD_NUM_PROGRAM_TARGETS);

static const bitfield32_t fallback_groups[] = {
	GPUS_FALLBACK_VERTEX_FOR_GROUPS,
	GPUS_FALLBACK_VERTEX_FOR_GROUPS,
	GPUS_FALLBACK_VERTEX_FOR_GROUPS,
	GPUS_FALLBACK_VERTEX_FOR_GROUPS,
	GPUS_FALLBACK_FRAGMENT_FOR_GROUPS,
};
ct_assert(sizeof(fallback_groups) / sizeof(fallback_groups[0]) == GLD_NUM_PROGRAM_TARGETS);

static const uint32_t bitfield_offsets[] = {
	GLD_VP_BITFIELD_OFFSET,
	GLD_CP_BITFIELD_OFFSET,
	GLD_EP_BITFIELD_OFFSET,
	GLD_GP_BITFIELD_OFFSET,
	GLD_FP_BITFIELD_OFFSET,
};
ct_assert(sizeof(bitfield_offsets) / sizeof(bitfield_offsets[0]) == GLD_NUM_PROGRAM_TARGETS);

ALWAYS_INLINE GLRVendBoundPipeProg glrUpdateProgramInline(GLDContext ctx, GLDProgram program_obj,
	GLDPipelineProgram pipeprog_obj, GLRCookie **program_cookie, bitfield32_t change3, uint32_t program_index)
{
	#pragma unused(change3)
	const GLDPipelineProgramData __attribute__((unused)) *pipeprog_data = pipeprog_obj->program_data;
	GLDPluginPipelineProgramData *plugin_pipeprog_data = pipeprog_obj->plugin_pipeprog_data;
	#if GLR_PIPEPROG_HAS_SHR_BUILD
	GLRVendShrPipeProg vendshrpipeprog;
	#endif
	#if GLR_PIPEPROG_HAS_CTX_BUILD
	GLRVendCtxPipeProg vendctxpipeprog;
	#endif
	bitfield32_t pipeprog_local_dirty_bits = plugin_pipeprog_data->pipeprog_dirty_bits;

	/* All context-independent processing of the program is handled first. */
	/* Clear the dirty bits for the program object, even if there aren't any.  Can't put this
	 ** within one of the branches below because of the masks. */
	plugin_pipeprog_data->pipeprog_dirty_bits = 0U;

	gpusAssert(pipeprog_data->data || pipeprog_data->new_ir_data);
	gpusAssert(pipeprog_data->target_index == program_index);

	gpusAssert(program_index < GLD_NUM_PROGRAM_TARGETS);
	#if GLR_PIPEPROG_HAS_CTX_BUILD
	bitfield32_t ctx_dirty_mask = ctx_dirty_masks[program_index];
	#endif
	#if GLR_PIPEPROG_HAS_SHR_BUILD
	bitfield32_t shr_dirty_mask = shr_dirty_masks[program_index];
	#endif

	#if GLR_PIPEPROG_HAS_SHR_BUILD
	/* All work associated with updating the context-independent program is
	 ** constrained by the dirty bits and done here. */
	/* Get the vendor program that may have already been built, but could be NULL. */
	vendshrpipeprog = pipeprog_obj->vendpipeprog;

	if(pipeprog_local_dirty_bits & shr_dirty_mask)
	{
		/* Update this program, with no consideration of the needs of this context.  The
		 ** driver is free to return pipeprog_obj->vendpipeprog, or create a new one.  It is
		 ** free to return pipeprog_obj->vendpipeprog even if the program's reference count
		 ** is greater than 1, if doing so will not crash other contexts that are currently
		 ** using the vendor program.  If we have no program data we don't call UpdateShrSys
		 ** because all it could do it release resources from a previous update, and that
		 ** is done here. */
		vendshrpipeprog = glrUpdateShrSysProgram[program_index](ctx->shared, program_obj,
			pipeprog_obj, vendshrpipeprog,
			pipeprog_local_dirty_bits);

		/* Deal with an object change. */
		if(vendshrpipeprog != pipeprog_obj->vendpipeprog)
		{
			if(vendshrpipeprog)
				glrRetainVendShrPipeProg(ctx->shared, vendshrpipeprog);
			if(pipeprog_obj->vendpipeprog)
				glrReleaseVendShrPipeProg(ctx->shared, pipeprog_obj->vendpipeprog);
			pipeprog_obj->vendpipeprog = vendshrpipeprog;
		}
	}
	#endif // GLR_PIPEPROG_HAS_SHR_BUILD

	/* Update the cookie with whatever bits are dirty. */
	glrSetCookieWithBits(ctx->shared, &pipeprog_obj->pipeprog_cookie,
						 pipeprog_local_dirty_bits);

	#if GLR_PIPEPROG_HAS_SHR_BUILD
	if(!vendshrpipeprog)
	{
		#if GLR_PIPEPROG_HAS_CTX_BUILD
		/* The returned ctx-specific programs is also NULL. */
		vendctxpipeprog = NULL;
		#endif
		/* The null cookie is returned. */
		*program_cookie = &ctx->shared->shr_null_cookie;
	}
	else
	#endif /* GLR_PIPEPROG_HAS_SHR_BUILD */
	{
		#if GLR_PIPEPROG_HAS_CTX_BUILD
		GLRProgramCacheElement *cache_prog;

		/* Find, or create, a cached program for this context. */
		cache_prog = gpumGetCachedProgram(ctx, pipeprog_obj);

		if(ctx_dirty_mask & GLD_PROGRAM_TEXTURES_BIT)
		{
			/* Merge the texture cookie from the context into the other program cookies. */
			pipeprog_obj->pipeprog_cookie.textures_cookie = ctx->textures_atom[program_index];
		}

		#if (GLR_PIPEPROG_ALL_CTX_DIRTY_MASK & GLD_PROGRAM_STATE_BIT)
		if(ctx_dirty_mask & GLD_PROGRAM_STATE_BIT)
		{
			/* Update the state cookie. */
			if(change3 & GLD_STATE3_VERTEX_PROGRAM_STATE << program_index)
			{
				pipeprog_obj->pipeprog_cookie.state_cookie = ++ctx->shared->shr_state_cookie;
			}
			else
			{
				pipeprog_obj->pipeprog_cookie.state_cookie = cache_prog->cacheprog_cookie.state_cookie;
			}
		}
		#endif // (GLR_PIPEPROG_ALL_CTX_DIRTY_MASK & GLD_PROGRAM_STATE_BIT)

		/* Produce new dirty bits to pass forward based on the cookie difference. */
		pipeprog_local_dirty_bits = glrGetCookieDiffBits(&pipeprog_obj->pipeprog_cookie,
														 &cache_prog->cacheprog_cookie);

		/* Get the vendor program that may have already been built, but could be NULL. */
		vendctxpipeprog = cache_prog->vendpipeprog;

		if(vendctxpipeprog)
		{
			#if GLR_PIPEPROG_CTX_TEST_STATE_CHANGE
			if(ctx_dirty_mask & GLD_PROGRAM_TEXTURES_BIT)
			{
				if((pipeprog_local_dirty_bits & GLR_PIPEPROG_VERT_CTX_DIRTY_MASK) ==
				   GLD_PROGRAM_TEXTURES_BIT)
				{
					// If the only dirty bit is the context state, ask the driver whether or
					// not a rebuild is really required
					if(!glrTestCtxSysProgram[program_index](ctx, program_obj, pipeprog_obj,
						#if GLR_PIPEPROG_HAS_SHR_BUILD
						vendshrpipeprog,
						#endif
						vendctxpipeprog))
					{
						// If a rebuild is not required, remove the textures bit and
						// make the cookie match the context
						pipeprog_obj->pipeprog_cookie.textures_cookie =
						ctx->ctx_prog_cookie[program_index].textures_cookie;
						pipeprog_local_dirty_bits &= ~GLD_PROGRAM_TEXTURES_BIT;
					}
				}
			}
			#endif // GLR_PIPEPROG_CTX_TEST_STATE_CHANGE
		}
		else
		{
			if(GLR_PIPEPROG_CTX_FAIL_FOR_CTX_STATE)
			{
				if(pipeprog_local_dirty_bits & GLD_PROGRAM_TEXTURES_BIT)
					pipeprog_local_dirty_bits |= GLD_PROGRAM_DATA_BIT;
				pipeprog_local_dirty_bits &= ~GLD_PROGRAM_STATE_BIT;
			}
			else if(ctx_dirty_mask & (GLD_PROGRAM_TEXTURES_BIT | GLD_PROGRAM_STATE_BIT))
			{
				pipeprog_local_dirty_bits &= ~(GLD_PROGRAM_TEXTURES_BIT | GLD_PROGRAM_STATE_BIT);
			}
		}

		/* All work associated with updating the context-dependent program is
		 ** constrained by the dirty bits and done here. */
		if(pipeprog_local_dirty_bits & ctx_dirty_mask)
		{
			/* Update this program considering of the needs of this context.  The
			 ** driver is free to return cache_prog->vendpipeprog, or create a new one.
			 ** If we have no program data we don't call UpdateCtxSys because all it could do it
			 ** release resources from a previous update, and that is done here. */
			vendctxpipeprog = glrUpdateCtxSysProgram[program_index](ctx, program_obj, pipeprog_obj,
													 #if GLR_PIPEPROG_HAS_SHR_BUILD
													 vendshrpipeprog,
													 #endif
													 vendctxpipeprog, pipeprog_local_dirty_bits);

			/* Deal with an object change. */
			if(vendctxpipeprog != cache_prog->vendpipeprog)
			{
				if(vendctxpipeprog)
					glrRetainVendCtxPipeProg(ctx, vendctxpipeprog);
				if(cache_prog->vendpipeprog)
					glrReleaseVendCtxPipeProg(ctx, cache_prog->vendpipeprog);
				cache_prog->vendpipeprog = vendctxpipeprog;
			}

			#if (GLR_PIPEPROG_ALL_CTX_DIRTY_MASK & GLD_PROGRAM_ALL_LOCALS_BIT)
			if (ctx_dirty_mask & GLD_PROGRAM_ALL_LOCALS_BIT)
			{
				if((pipeprog_local_dirty_bits & (GLD_PROGRAM_LOCALS_BIT | GLD_PROGRAM_ALL_LOCALS_BIT)) == GLD_PROGRAM_LOCALS_BIT)
					pipeprog_obj->pipeprog_cookie.all_locals_cookie = ++ctx->shared->shr_all_locals_cookie;
			}
			#endif // (GLR_PIPEPROG_ALL_CTX_DIRTY_MASK & GLD_PROGRAM_ALL_LOCALS_BIT)
		}

		/* Structure copy. */
		cache_prog->cacheprog_cookie = pipeprog_obj->pipeprog_cookie;

		if(vendctxpipeprog)
		/* Return the cookie of the cached program. */
			*program_cookie = &cache_prog->cacheprog_cookie;
		else
		/* Return the null cookie. */
			*program_cookie = &ctx->shared->shr_null_cookie;

		#else /* !GLR_PIPEPROG_HAS_CTX_BUILD */

		/* Return the main cookie of the shared program. */
		*program_cookie = &pipeprog_obj->pipeprog_cookie;

		#endif /* !GLR_PIPEPROG_HAS_CTX_BUILD */
	}

	#if GLR_PIPEPROG_HAS_CTX_BUILD
	return vendctxpipeprog;
	#else
	return vendshrpipeprog;
	#endif
}

#if GLR_HAS_VERTEX_SHADER

static GLRVendBoundPipeProg glrUpdateVertexProgramInline(
	GLDContext ctx, GLDProgram program_obj, GLDPipelineProgram pipeprog_obj,
	GLRCookie **program_cookie, bitfield32_t change3)
{
	return glrUpdateProgramInline(ctx, program_obj, pipeprog_obj,
		program_cookie, change3, GLD_PROGRAM_INDEX_VERTEX);
}

#else // if !GLR_HAS_VERTEX_SHADER

ALWAYS_INLINE void glrUpdateVertexProgramInline(
	GLDContext __attribute__((unused)) ctx, GLDProgram __attribute__((unused)) program_obj, GLDPipelineProgram __attribute__((unused)) pipeprog_obj,
	GLRCookie __attribute__((unused)) **program_cookie, bitfield32_t __attribute__((unused)) change3)
{
	//Define nothing here.  Optimizer will remove branches calling this stub
}

#endif // !GLR_HAS_VERTEX_SHADER

static GLRVendBoundPipeProg glrUpdateFragmentProgramInline(
	GLDContext ctx, GLDProgram program_obj, GLDPipelineProgram pipeprog_obj,
	GLRCookie **program_cookie, bitfield32_t change3)
{
	return glrUpdateProgramInline(ctx, program_obj, pipeprog_obj,
	   program_cookie, change3, GLD_PROGRAM_INDEX_FRAGMENT);
}

/* Pipeline Programs */
enum32_t gldCreatePipelineProgram(GLDShareGroup shared, GLDPipelineProgram *pipprg_ret,
	const GLDPipelineProgramData *data, GLDPluginPipelineProgramData *plugin_data)
{
	GLDPipelineProgram pipprg;

	pipprg = (GLDPipelineProgram) gpusCalloc(1,
		shared->io_data->struct_sizes.pipeline_program);

	pipprg->program_data = data;
	pipprg->plugin_pipeprog_data = plugin_data;

	gpumSetCookieNoProgram(&pipprg->pipeprog_cookie);
	#if GLR_PIPEPROG_HAS_SHR_BUILD
	gpusAssert(pipprg->vendpipeprog == NULL);
	#endif // GLR_PIPEPROG_HAS_SHR_BUILD
	#if GLR_PIPEPROG_HAS_CTX_BUILD
	gpusAssert(pipprg->cache_first == NULL);
	gpusAssert(pipprg->cache_last == NULL);
	gpusAssert(pipprg->cache_size == 0);
	gpusAssert(!pipprg->prev_in_shared_list);

	// Add program to shared list.
	gpusSharedMutexLock(shared);

	if(shared->program_list)
		shared->program_list->prev_in_shared_list = pipprg;
	pipprg->next_in_shared_list = shared->program_list;
	shared->program_list = pipprg;

	gpusSharedMutexUnlock(shared);
	#endif // GLR_PIPEPROG_HAS_CTX_BUILD

	*pipprg_ret = pipprg;

	return GLD_NO_ERROR;
}


enum32_t gldModifyPipelineProgram(GLDShareGroup shared, GLDPipelineProgram pipprg,
	bitfield32_t dirty)
{
	glrModifySysPipelineProgram(shared, pipprg, dirty);
	return GLD_NO_ERROR;
}

enum32_t gldGetPipelineProgramInfo(GLDContext ctx, GLDPipelineProgram pipeprog_obj,
	enum32_t pname, int32_t *param)
{
	GLDShareGroup shared = ctx->shared;
	GLRVendBoundPipeProg vendpipeprog;
	GLRCookie *program_cookie;
	uint8_t target_index = pipeprog_obj->program_data->target_index;

	// Don't ask the driver for info on an empty program.
	if(!pipeprog_obj->program_data->data && !pipeprog_obj->program_data->new_ir_data)
	{
		*param = 0;
		return GLD_NO_ERROR;
	}

	gpusSharedMutexLock(shared);

	#if GLR_PIPEPROG_HAS_CTX_BUILD
	glrUpdateSysTexturesForCtxProgram(ctx, pipeprog_obj);
	#endif

	vendpipeprog = glrUpdateProgramInline(ctx, NULL, pipeprog_obj, &program_cookie, 0U, target_index);

	gpusSharedMutexUnlock(shared);

	if(vendpipeprog)
		glrGetSysPipelineProgramInfo(
			#if GLR_PIPEPROG_HAS_CTX_BUILD
			ctx,
			#else
			shared,
			#endif
			pipeprog_obj, vendpipeprog, pname, param);
	else
		*param = 0;

	return GLD_NO_ERROR;
}

enum32_t gldDestroyPipelineProgram(GLDShareGroup shared, GLDPipelineProgram pipprg)
{
	#if GLR_PIPEPROG_HAS_SHR_BUILD
	if(pipprg->vendpipeprog)
	{
		glrReleaseVendShrPipeProg(shared, pipprg->vendpipeprog);
		pipprg->vendpipeprog = NULL;
	}
	#endif
	#if GLR_PIPEPROG_HAS_CTX_BUILD
	gpusSharedMutexLock(shared);

	while(pipprg->cache_size)
		gpumDeleteCachedProgram(shared, pipprg, pipprg->cache_first);
	gpusAssert(pipprg->cache_first == NULL);
	gpusAssert(pipprg->cache_last == NULL);

	// Remove program from shared list.
	if(shared->program_list == pipprg)
		shared->program_list = pipprg->next_in_shared_list;
	if(pipprg->next_in_shared_list)
		pipprg->next_in_shared_list->prev_in_shared_list = pipprg->prev_in_shared_list;
	if(pipprg->prev_in_shared_list)
		pipprg->prev_in_shared_list->next_in_shared_list = pipprg->next_in_shared_list;

	gpusSharedMutexUnlock(shared);
	#endif

	glrDeleteSysPipelineProgram(shared, pipprg);
	gpusFree(pipprg);

	return GLD_NO_ERROR;
}

void gldUnbindPipelineProgram(GLDContext ctx, GLDPipelineProgram pipprg)
{
	uint32_t i;
	for(i = 0; i < GLD_NUM_PROGRAM_TARGETS; i++)
	{
		if(ctx->cur_pipeline_program[i] == pipprg)
		{
			gpusAssert(pipprg->program_data->target_index == i);

			bitfield32_t pipeprog_dirty_mask = bind_dirty_masks[i];

			if(ctx->prog_vend[i])
			{
				gpusSharedMutexLock(ctx->shared);
				glrBindSysProgram[i](ctx, NULL, NULL, NULL, NULL,
									 pipeprog_dirty_mask);
				#if GLR_PIPEPROG_HAS_CTX_BUILD
				glrReleaseVendCtxPipeProg(ctx, ctx->prog_vend[i]);
				#else
				glrReleaseVendShrPipeProg(ctx->shared, ctx->prog_vend[i]);
				#endif
				gpumSetCookieNoProgram(&ctx->ctx_prog_cookie[i]);
				ctx->prog_vend[i] = NULL;
				gpusSharedMutexUnlock(ctx->shared);
			}

			ctx->cur_pipeline_program[i] = NULL;
			break;
		}
	}
}

ALWAYS_INLINE void glrLoadCurrentProgram(uint32_t program_index, GLDContext ctx, GLDProgram program_obj,
	GLDPipelineProgram pipeprog_obj, bitfield32_t *change_array)
{
	GLRVendBoundPipeProg vendpipeprog;
	GLRCookie *program_cookie;
	bitfield32_t pipeprog_local_dirty_bits;

	gpusAssert(program_index < GLD_NUM_PROGRAM_TARGETS);
	bitfield32_t bind_dirty_mask = bind_dirty_masks[program_index];
	bitfield32_t ctx_dirty_mask = ctx_dirty_masks[program_index];
	bitfield32_t shr_dirty_mask = shr_dirty_masks[program_index];
	#if GLR_PIPEPROG_NEEDS_DISABLE_NOTIFIER
	bitfield32_t fallback_mask = fallback_masks[program_index];
	bitfield32_t fallback_group = fallback_groups[program_index];
	#endif
	bitfield32_t fallback_bit = fallback_bits[program_index];
	uint32_t bitfield_offset = bitfield_offsets[program_index];

	/* All context-independent processing of the program is handled first. */

	if(pipeprog_obj)
	{
		vendpipeprog = glrUpdateProgramInline(ctx, program_obj, pipeprog_obj,
			&program_cookie, change_array[3], program_index);

		/* Produce new dirty bits to pass forward based on the cookie difference. */
		pipeprog_local_dirty_bits = glrGetCookieDiffBits(program_cookie, &ctx->ctx_prog_cookie[program_index]);

		/* Copy the program's cookie to the context. */
		ctx->ctx_prog_cookie[program_index] = *program_cookie;

		/* No vendor prog, so no pipeline prog. */
		if(!vendpipeprog)
		{
			#if GLR_PIPEPROG_NEEDS_DISABLE_NOTIFIER
			bitfield32_t old_code = ctx->disable_code;
			#endif
			/* A dataless program indicates that the program has no program text, or it
			** has program text with a syntax error.  GLEngine will generate INVALID_OP
			** if the user does anything that requires vertex transformation.  If that
			** fails and we somehow end up trying to transform vertices in the driver we
			** should assert. */
			ctx->disable_code |= fallback_bit;

			program_obj  = NULL;
			pipeprog_obj = NULL;
			#if GLR_PIPEPROG_NEEDS_DISABLE_NOTIFIER
			glrUpdateForDisableCodeChange(ctx, old_code);
			#endif
		}
		else if(((bind_dirty_mask & GLD_PROGRAM_STATE_BIT) &&
				!(ctx_dirty_mask & GLD_PROGRAM_STATE_BIT)) &&
				(change_array[3] & (GLD_STATE3_VERTEX_PROGRAM_STATE << program_index)))
		{
			pipeprog_local_dirty_bits |= GLD_PROGRAM_STATE_BIT;
		}
	}
	else
	{
		/* Disable program mode and use fixed function. */
		vendpipeprog = NULL;

		/* If the previous program was non-NULL, this will cause a program bind to NULL. */
		if(!gpumIsCookieNoProgram(&ctx->ctx_prog_cookie[program_index]))
		{
			pipeprog_local_dirty_bits = GLD_PROGRAM_ALL_BITS;

			/* Set the context's cookie to match this program. */
			gpumSetCookieNoProgram(&ctx->ctx_prog_cookie[program_index]);
		}
		else
		{
			pipeprog_local_dirty_bits = 0U;
		}
	}

	#if GLR_PIPEPROG_NEEDS_DISABLE_NOTIFIER
	if(program_index != GLD_PROGRAM_INDEX_GEOMETRY)
	{
		if((ctx->disable_code & fallback_mask) == fallback_group)
		{
			program_obj  = NULL;
			pipeprog_obj = NULL;
			if(vendpipeprog)
			{
				#if GLR_PIPEPROG_HAS_CTX_BUILD
				glrRetainVendCtxPipeProg(ctx, vendpipeprog);
				glrReleaseVendCtxPipeProg(ctx, vendpipeprog);
				#else
				glrRetainVendShrPipeProg(ctx->shared, vendpipeprog);
				glrReleaseVendShrPipeProg(ctx->shared, vendpipeprog);
				#endif
			}
			vendpipeprog = NULL;
		}
		gpusAssert(!(ctx->disable_code & fallback_mask) || !pipeprog_obj);
	}
	#endif

	/* All context-specific processing of the program is handled after this point. */

	/* test if anything about this program has changed with respect to this context. */
	if(pipeprog_local_dirty_bits & (
		shr_dirty_mask |
		ctx_dirty_mask |
		bind_dirty_mask))
	{
		/* Bind the program to this context. */
		if(pipeprog_local_dirty_bits & bind_dirty_mask)
		{
			glrBindSysProgram[program_index](ctx, program_obj, pipeprog_obj, vendpipeprog,
				change_array + bitfield_offset, pipeprog_local_dirty_bits);
		}

		/* Deal with an object change. */
		if(vendpipeprog != ctx->prog_vend[program_index])
		{
			if(vendpipeprog)
			{
				#if GLR_PIPEPROG_HAS_CTX_BUILD
				glrRetainVendCtxPipeProg(ctx, vendpipeprog);
				#else
				glrRetainVendShrPipeProg(ctx->shared, vendpipeprog);
				#endif
			}
			if(ctx->prog_vend[program_index])
			{
				#if GLR_PIPEPROG_HAS_CTX_BUILD
				glrReleaseVendCtxPipeProg(ctx, ctx->prog_vend[program_index]);
				#else
				glrReleaseVendShrPipeProg(ctx->shared, ctx->prog_vend[program_index]);
				#endif
			}
			ctx->prog_vend[program_index] = vendpipeprog;
		}

		gpusAssert(vendpipeprog || !pipeprog_obj);
	}

	ctx->cur_pipeline_program[program_index] = pipeprog_obj;
}

#if GLR_HAS_VERTEX_SHADER
static void glrLoadCurrentVertexProgram(GLDContext __attribute__((unused)) ctx, GLDProgram __attribute__((unused)) program_obj,
	GLDPipelineProgram __attribute__((unused)) pipeprog_obj, bitfield32_t __attribute__((unused)) *change_array)
{
	glrLoadCurrentProgram(GLD_PROGRAM_INDEX_VERTEX, ctx, program_obj, pipeprog_obj, change_array);
}
#else // if !GLR_HAS_VERTEX_SHADER
ALWAYS_INLINE void glrLoadCurrentVertexProgram(GLDContext ctx, GLDProgram program_obj,
	GLDPipelineProgram pipeprog_obj, bitfield32_t *change_array)
{
}
#endif // !GLR_HAS_VERTEX_SHADER


#if GLR_HAS_TESSELLATION_SHADER
static void glrLoadCurrentTessControlProgram(GLDContext ctx, GLDProgram program_obj,
	GLDPipelineProgram pipeprog_obj, bitfield32_t *change_array)
{
	glrLoadCurrentProgram(GLD_PROGRAM_INDEX_TESS_CONTROL, ctx, program_obj, pipeprog_obj, change_array);
}

static void glrLoadCurrentTessEvaluationProgram(GLDContext ctx, GLDProgram program_obj,
	GLDPipelineProgram pipeprog_obj, bitfield32_t *change_array)
{
	glrLoadCurrentProgram(GLD_PROGRAM_INDEX_TESS_EVALUATION, ctx, program_obj, pipeprog_obj, change_array);
}

#else // if !GLR_HAS_GEOMETRY_SHADER
ALWAYS_INLINE void glrLoadCurrentTessControlProgram(GLDContext __attribute__((unused)) ctx, GLDProgram __attribute__((unused)) program_obj,
	GLDPipelineProgram __attribute__((unused)) pipeprog_obj, bitfield32_t __attribute__((unused)) *change_array)
{
}
ALWAYS_INLINE void glrLoadCurrentTessEvaluationProgram(GLDContext __attribute__((unused)) ctx, GLDProgram __attribute__((unused)) program_obj,
	GLDPipelineProgram __attribute__((unused)) pipeprog_obj, bitfield32_t __attribute__((unused)) *change_array)
{
}
#endif // !GLR_HAS_GEOMETRY_SHADER

#if GLR_HAS_GEOMETRY_SHADER
static void glrLoadCurrentGeometryProgram(GLDContext ctx, GLDProgram program_obj,
	GLDPipelineProgram pipeprog_obj, bitfield32_t *change_array)
{
	glrLoadCurrentProgram(GLD_PROGRAM_INDEX_GEOMETRY, ctx, program_obj, pipeprog_obj, change_array);
}

#else // if !GLR_HAS_GEOMETRY_SHADER
ALWAYS_INLINE void glrLoadCurrentGeometryProgram(GLDContext __attribute__((unused)) ctx, GLDProgram __attribute__((unused)) program_obj,
	GLDPipelineProgram __attribute__((unused)) pipeprog_obj, bitfield32_t __attribute__((unused)) *change_array)
{
}
#endif // !GLR_HAS_GEOMETRY_SHADER


static void glrLoadCurrentFragmentProgram(GLDContext ctx, GLDProgram program_obj,
	GLDPipelineProgram pipeprog_obj, bitfield32_t *change_array)
{
	glrLoadCurrentProgram(GLD_PROGRAM_INDEX_FRAGMENT, ctx, program_obj, pipeprog_obj, change_array);
}

bool32_t gpusHandleProcessingModeChange(GLDContext ctx)
{
	bitfield32_t newBits, maskBits;

	newBits = ctx->forced_disable_code;

	gpusAssert(!(newBits &
		~(GPUS_FALLBACK_VERTEX_FOR_DRIVER_REQUEST | GPUS_FALLBACK_FRAGMENT_FOR_DRIVER_REQUEST)));

	#if !GLR_HAS_VERTEX_SHADER
	newBits &= ~GPUS_FALLBACK_VERTEX_FOR_DRIVER_REQUEST;
	#endif

	// This code allows GLEngine to force the driver into the vertex or fragment
    // fallback paths without actually using an unsupported state vector.
	maskBits = (
		#if GLR_HAS_VERTEX_SHADER
		GPUS_FALLBACK_VERTEX_FOR_DRIVER_REQUEST |
		GPUS_FALLBACK_VERTEX_FOR_PROCESS_MODE |
		#endif
		GPUS_FALLBACK_FRAGMENT_FOR_DRIVER_REQUEST |
		GPUS_FALLBACK_FRAGMENT_FOR_PROCESS_MODE);

	switch(ctx->state->processing_mode)
	{
	#if GLR_HAS_VERTEX_SHADER
	case GLD_FULL_TRANSFORM:
		break;
	#endif
	case GLD_POST_VERTEX:
		#if GLR_HAS_VERTEX_SHADER
		newBits |= GPUS_FALLBACK_VERTEX_FOR_PROCESS_MODE;
		#endif
		break;
	case GLD_POST_FRAGMENT:
	default:
		gpusAssert(ctx->state->processing_mode == GLD_POST_FRAGMENT);
		#if GLR_HAS_VERTEX_SHADER
		newBits |= GPUS_FALLBACK_VERTEX_FOR_PROCESS_MODE | GPUS_FALLBACK_FRAGMENT_FOR_PROCESS_MODE;
		#else
		newBits |= GPUS_FALLBACK_FRAGMENT_FOR_PROCESS_MODE;
		#endif
		break;
	}

	if(newBits ^ (ctx->disable_code & maskBits))
	{
		ctx->disable_code &= ~maskBits;
		ctx->disable_code |= newBits;
		return true;
	}
	else
	{
		return false;
	}
}

void gpusLoadCurrentPipelinePrograms(GLDContext ctx, bitfield32_t *change_array)
{
	GLDPipelineProgram pipeprog_obj;
	GLDProgram program_obj;
	#if GLR_PIPEPROG_NEEDS_DISABLE_NOTIFIER
	bitfield32_t old_code;
	#endif /* GLR_PIPEPROG_NEEDS_DISABLE_NOTIFIER */

	bool32_t glsl_disabled = false;

	/* Handle program-level GLSL stuff. */
	if(change_array[3] & GLD_STATE3_UNIFORM_BUFFER_BIND)
	{
		if(!gpumUpdateUniformBuffers(ctx))
			glsl_disabled = true;
	}

	#if GLR_HAS_VERTEX_SHADER
	#if 0
	// TODO: Write a test case that proves this bug is real - then enable this code.
	// If the fragment stage is being rebuilt but the vertex stage is not - and we're already
	// in fragment fallback mode because of a failure to compile the fragment stage, we must
	// reset the vertex stage first or we'll end up with the fragment stage in normal mode
	// but the vertex stage in fallback.
	if(change_array[3] & (
		#if GLR_PIPEPROG_FRAG_CTX_DIRTY_MASK & GLD_PROGRAM_STATE_BIT
		GLD_STATE3_FRAGMENT_PROGRAM_STATE |
		#endif
		GLD_STATE3_FRAGMENT_PROGRAM_BIND))
	{
		if(!(change_array[3] & (
				#if GLR_PIPEPROG_VERT_CTX_DIRTY_MASK & GLD_PROGRAM_STATE_BIT
				GLD_STATE3_VERTEX_PROGRAM_STATE |
				#endif
				GLD_STATE3_VERTEX_PROGRAM_BIND |
				#if !GLR_HAS_GEOMETRY_SHADER
				GLD_STATE3_GEOMETRY_PROGRAM_BIND |
				#endif
				GLD_STATE3_PROGRAM_OBJECT_BIND |
				GLD_STATE3_PROCESSING_MODE)) &&
			(ctx->disable_code & GPUS_FALLBACK_FRAGMENT_FOR_FRAG_PROG))
		{
			change_array[3] |= GLD_STATE3_VERTEX_PROGRAM_BIND;
		}
	}
	#endif

	/* Handle Vertex Programs/Shaders */
	if(change_array[3] & (
		#if GLR_PIPEPROG_VERT_CTX_DIRTY_MASK & GLD_PROGRAM_STATE_BIT
		GLD_STATE3_VERTEX_PROGRAM_STATE |
		#endif
		GLD_STATE3_PROGRAM_OBJECT_BIND |
		GLD_STATE3_VERTEX_PROGRAM_BIND |
		#if !GLR_HAS_TESSELLATION_SHADER
		GLD_STATE3_TESS_CONTROL_PROGRAM_BIND |
		GLD_STATE3_TESS_EVALUATION_PROGRAM_BIND |
		#endif
		#if !GLR_HAS_GEOMETRY_SHADER
		GLD_STATE3_GEOMETRY_PROGRAM_BIND |
		#endif
		GLD_STATE3_PROCESSING_MODE))
	{
		#if GLR_PIPEPROG_NEEDS_DISABLE_NOTIFIER
		old_code = ctx->disable_code;
		#endif /* GLR_PIPEPROG_NEEDS_DISABLE_NOTIFIER */
		ctx->disable_code &= ~(
			GPUS_FALLBACK_VERTEX_FOR_VERT_PROG |
			GPUS_FALLBACK_VERTEX_FOR_VERT_BUF_REQ |
			#if !GLR_HAS_TESSELLATION_SHADER
			GPUS_FALLBACK_VERTEX_FOR_CTRL_PROG |
			GPUS_FALLBACK_VERTEX_FOR_CTRL_BUF_REQ |
			GPUS_FALLBACK_VERTEX_FOR_EVAL_PROG |
			GPUS_FALLBACK_VERTEX_FOR_EVAL_BUF_REQ |
			#endif
			#if !GLR_HAS_GEOMETRY_SHADER
			GPUS_FALLBACK_VERTEX_FOR_GEOM_PROG |
			GPUS_FALLBACK_VERTEX_FOR_GEOM_BUF_REQ |
			#endif
			GPUS_FALLBACK_FRAGMENT_FOR_VERTEX_FALLBACK);

		program_obj = ctx->plugin_state->programs[GLD_PROGRAM_INDEX_VERTEX];

		/* Check if the vertex stage is disabled. */
		if(ctx->disable_code & (
			GPUS_FALLBACK_VERTEX_EXTERNAL_MASK |
			GPUS_FALLBACK_FRAGMENT_EXTERNAL_MASK))
		{
			program_obj = NULL;
			pipeprog_obj = NULL;
		}
		/* Check for GLSL */
		else if(program_obj && (program_obj->program_data->valid_shaders & GLD_SHADER_BIT_VERTEX))
		{
			/* Trying to use GLSL. */
			/* If a transform feedback varying present but the driver does not implement
			** transform feedback, fail to load vertex shader */
			if(glsl_disabled)
			{
				ctx->disable_code |= GPUS_FALLBACK_VERTEX_FOR_VERT_BUF_REQ;
				program_obj = NULL;
				pipeprog_obj = NULL;
			}
			#if !GLR_HAS_TESSELLATION_SHADER
			else if(program_obj->program_data->valid_shaders & GLD_SHADER_BIT_TESS_CONTROL)
			{
				ctx->disable_code |= GPUS_FALLBACK_VERTEX_FOR_CTRL_PROG;
				program_obj = NULL;
				pipeprog_obj = NULL;
			}
			else if(program_obj->program_data->valid_shaders & GLD_SHADER_BIT_TESS_EVALUATION)
			{
				ctx->disable_code |= GPUS_FALLBACK_VERTEX_FOR_EVAL_PROG;
				program_obj = NULL;
				pipeprog_obj = NULL;
			}
			#endif
			#if !GLR_HAS_GEOMETRY_SHADER
			else if(program_obj->program_data->valid_shaders & GLD_SHADER_BIT_GEOMETRY)
			{
				ctx->disable_code |= GPUS_FALLBACK_VERTEX_FOR_GEOM_PROG;
				program_obj = NULL;
				pipeprog_obj = NULL;
			}
			#endif
			else
			{
				pipeprog_obj = program_obj->plugin_program_data->pipeline_shaders[GLD_SHADER_INDEX_VERTEX];
				gpusAssert(pipeprog_obj);
				gpusAssert(pipeprog_obj->program_data->data ||  pipeprog_obj->program_data->new_ir_data);
			}
		}
		/* Check for ARB */
		else
		{
			program_obj = NULL;
			pipeprog_obj = ctx->plugin_state->pipeline_program[GLD_PROGRAM_INDEX_VERTEX];
			gpusAssert(pipeprog_obj);
			gpusAssert(pipeprog_obj->program_data->data || pipeprog_obj->program_data->new_ir_data);
		}

		#if GLR_PIPEPROG_NEEDS_DISABLE_NOTIFIER
		if(ctx->disable_code != old_code)
			glrUpdateForDisableCodeChange(ctx, old_code);
		#endif // GLR_PIPEPROG_NEEDS_DISABLE_NOTIFIER

		glrLoadCurrentVertexProgram(ctx, program_obj, pipeprog_obj, change_array);
	}
	#if (GLR_PIPEPROG_VERT_BIND_DIRTY_MASK & GLD_PROGRAM_STATE_BIT) && \
		!(GLR_PIPEPROG_VERT_CTX_DIRTY_MASK & GLD_PROGRAM_STATE_BIT)
	else if(change_array[3] & GLD_STATE3_VERTEX_PROGRAM_STATE)
	{
		if(ctx->cur_pipeline_program[GLD_PROGRAM_INDEX_VERTEX])
		{
			program_obj = ctx->plugin_state->programs[GLD_PROGRAM_INDEX_VERTEX];
			if(program_obj && !(program_obj->program_data->valid_shaders & GLD_SHADER_BIT_VERTEX))
				program_obj = NULL;

			glrBindSysVertexProgram(ctx,
				program_obj,
				ctx->cur_pipeline_program[GLD_PROGRAM_INDEX_VERTEX],
				ctx->prog_vend[GLD_PROGRAM_INDEX_VERTEX],
				change_array + GLD_VP_BITFIELD_OFFSET,
				GLD_PROGRAM_STATE_BIT);
		}
	}
	#endif
	#endif /* GLR_HAS_VERTEX_SHADER */


	#if GLR_HAS_TESSELLATION_SHADER
	/* Handle Tessellation Control Shaders */
	if(change_array[3] & (
		#if GLR_PIPEPROG_CTRL_CTX_DIRTY_MASK & GLD_PROGRAM_STATE_BIT
		GLD_STATE3_TESS_CONTROL_PROGRAM_STATE |
		#endif
		GLD_STATE3_PROGRAM_OBJECT_BIND |
		GLD_STATE3_TESS_CONTROL_PROGRAM_BIND |
		GLD_STATE3_PROCESSING_MODE))
	{
		#if GLR_PIPEPROG_NEEDS_DISABLE_NOTIFIER
		old_code = ctx->disable_code;
		#endif /* GLR_PIPEPROG_NEEDS_DISABLE_NOTIFIER */
		ctx->disable_code &= ~(
			GPUS_FALLBACK_VERTEX_FOR_CTRL_PROG |
			GPUS_FALLBACK_VERTEX_FOR_CTRL_BUF_REQ |
			GPUS_FALLBACK_FRAGMENT_FOR_VERTEX_FALLBACK);

		program_obj = ctx->plugin_state->programs[GLD_PROGRAM_INDEX_TESS_CONTROL];

		#if GLR_ASSERT
		if(program_obj)
		{
			gpusAssert(ctx->plugin_state->programs[GLD_PROGRAM_INDEX_VERTEX]); // if there is a tess-ctrl shader there must be a vertex shader
			gpusAssert(ctx->plugin_state->programs[GLD_PROGRAM_INDEX_VERTEX]->program_data->valid_shaders & GLD_SHADER_BIT_VERTEX); // the vertex shader must have a valid vertex shader
			gpusAssert(ctx->plugin_state->programs[GLD_PROGRAM_INDEX_TESS_CONTROL]->program_data->valid_shaders & GLD_SHADER_BIT_TESS_CONTROL); // the tess-ctrl shader must have a valid tess-ctrl shader
		}
		#endif

		/* Check if the vertex stage is disabled. */
		if(ctx->disable_code & (
			GPUS_FALLBACK_VERTEX_EXTERNAL_MASK |
			GPUS_FALLBACK_FRAGMENT_EXTERNAL_MASK))
		{
			program_obj = NULL;
			pipeprog_obj = NULL;
		}
		/* Check for GLSL */
		else if(program_obj && (program_obj->program_data->valid_shaders & GLD_SHADER_BIT_TESS_CONTROL))
		{
			/* Trying to use GLSL. */
			/* If a transform feedback varying present but the driver does not implement
			** transform feedback, fail to load vertex shader */
			if(glsl_disabled)
			{
				ctx->disable_code |= GPUS_FALLBACK_VERTEX_FOR_CTRL_BUF_REQ;
				program_obj = NULL;
				pipeprog_obj = NULL;
			}
			else
			{
				pipeprog_obj = program_obj->plugin_program_data->pipeline_shaders[GLD_SHADER_INDEX_TESS_CONTROL];
				gpusAssert(pipeprog_obj);
				gpusAssert(pipeprog_obj->program_data->data || pipeprog_obj->program_data->new_ir_data);
			}
		}
		/* No tessellation control shader is enabled */
		else
		{
			program_obj = NULL;
			pipeprog_obj = NULL;
		}

		#if GLR_PIPEPROG_NEEDS_DISABLE_NOTIFIER
		if(ctx->disable_code != old_code)
			glrUpdateForDisableCodeChange(ctx, old_code);
		#endif // GLR_PIPEPROG_NEEDS_DISABLE_NOTIFIER

		glrLoadCurrentTessControlProgram(ctx, program_obj, pipeprog_obj, change_array);
	}
	#if (GLR_PIPEPROG_CTRL_BIND_DIRTY_MASK & GLD_PROGRAM_STATE_BIT) && \
		!(GLR_PIPEPROG_CTRL_CTX_DIRTY_MASK & GLD_PROGRAM_STATE_BIT)
	else if(change_array[3] & GLD_STATE3_TESS_CONTROL_PROGRAM_STATE)
	{
		if(ctx->cur_pipeline_program[GLD_PROGRAM_INDEX_TESS_CONTROL])
		{
			program_obj = ctx->plugin_state->programs[GLD_PROGRAM_INDEX_TESS_CONTROL];
			gpusAssert(program_obj &&
					   (program_obj->program_data->valid_shaders & GLD_SHADER_BIT_TESS_CONTROL));

			glrBindSysTessControlProgram(ctx,
				program_obj,
				ctx->cur_pipeline_program[GLD_PROGRAM_INDEX_TESS_CONTROL],
				ctx->prog_vend[GLD_PROGRAM_INDEX_TESS_CONTROL],
				change_array + GLD_CP_BITFIELD_OFFSET,
				GLD_PROGRAM_STATE_BIT);
		}
	}
	#endif

	/* Handle Tessellation Evaluation Shaders */
	if(change_array[3] & (
		#if GLR_PIPEPROG_EVAL_CTX_DIRTY_MASK & GLD_PROGRAM_STATE_BIT
		GLD_STATE3_TESS_EVALUATION_PROGRAM_STATE |
		#endif
		GLD_STATE3_PROGRAM_OBJECT_BIND |
		GLD_STATE3_TESS_EVALUATION_PROGRAM_BIND |
		GLD_STATE3_PROCESSING_MODE))
	{
		#if GLR_PIPEPROG_NEEDS_DISABLE_NOTIFIER
		old_code = ctx->disable_code;
		#endif /* GLR_PIPEPROG_NEEDS_DISABLE_NOTIFIER */
		ctx->disable_code &= ~(
			GPUS_FALLBACK_VERTEX_FOR_EVAL_PROG |
			GPUS_FALLBACK_VERTEX_FOR_EVAL_BUF_REQ |
			GPUS_FALLBACK_FRAGMENT_FOR_VERTEX_FALLBACK);

		program_obj = ctx->plugin_state->programs[GLD_PROGRAM_INDEX_TESS_EVALUATION];

		#if GLR_ASSERT
		if(program_obj)
		{
			gpusAssert(ctx->plugin_state->programs[GLD_PROGRAM_INDEX_VERTEX]); // if there is a tess-eval shader there must be a vertex shader
			gpusAssert(ctx->plugin_state->programs[GLD_PROGRAM_INDEX_VERTEX]->program_data->valid_shaders & GLD_SHADER_BIT_VERTEX); // the vertex shader must have a valid vertex shader
			gpusAssert(ctx->plugin_state->programs[GLD_PROGRAM_INDEX_TESS_EVALUATION]->program_data->valid_shaders & GLD_SHADER_BIT_TESS_EVALUATION); // the tess-eval shader must have a valid tess-eval shader
		}
		#endif

		/* Check if the vertex stage is disabled. */
		if(ctx->disable_code & (
			GPUS_FALLBACK_VERTEX_EXTERNAL_MASK |
			GPUS_FALLBACK_FRAGMENT_EXTERNAL_MASK))
		{
			program_obj = NULL;
			pipeprog_obj = NULL;
		}
		/* Check for GLSL */
		else if(program_obj && (program_obj->program_data->valid_shaders & GLD_SHADER_BIT_TESS_EVALUATION))
		{
			/* Trying to use GLSL. */
			/* If a transform feedback varying present but the driver does not implement
			** transform feedback, fail to load vertex shader */
			if(glsl_disabled)
			{
				ctx->disable_code |= GPUS_FALLBACK_VERTEX_FOR_EVAL_BUF_REQ;
				program_obj = NULL;
				pipeprog_obj = NULL;
			}
			else
			{
				pipeprog_obj = program_obj->plugin_program_data->pipeline_shaders[GLD_SHADER_INDEX_TESS_EVALUATION];
				gpusAssert(pipeprog_obj);
				gpusAssert(pipeprog_obj->program_data->data || pipeprog_obj->program_data->new_ir_data);
			}
		}
		/* No tessellation evaluation shader is enabled */
		else
		{
			program_obj = NULL;
			pipeprog_obj = NULL;
		}

		#if GLR_PIPEPROG_NEEDS_DISABLE_NOTIFIER
		if(ctx->disable_code != old_code)
			glrUpdateForDisableCodeChange(ctx, old_code);
		#endif // GLR_PIPEPROG_NEEDS_DISABLE_NOTIFIER

		glrLoadCurrentTessEvaluationProgram(ctx, program_obj, pipeprog_obj, change_array);
	}
	#if (GLR_PIPEPROG_EVAL_BIND_DIRTY_MASK & GLD_PROGRAM_STATE_BIT) && \
		!(GLR_PIPEPROG_EVAL_CTX_DIRTY_MASK & GLD_PROGRAM_STATE_BIT)
	else if(change_array[3] & GLD_STATE3_TESS_EVALUATION_PROGRAM_STATE)
	{
		if(ctx->cur_pipeline_program[GLD_PROGRAM_INDEX_TESS_EVALUATION])
		{
			program_obj = ctx->plugin_state->programs[GLD_PROGRAM_INDEX_TESS_EVALUATION];
			gpusAssert(program_obj &&
					   (program_obj->program_data->valid_shaders & GLD_SHADER_BIT_TESS_EVALUATION));

			glrBindSysTessEvaluationProgram(ctx,
				program_obj,
				ctx->cur_pipeline_program[GLD_PROGRAM_INDEX_TESS_EVALUATION],
				ctx->prog_vend[GLD_PROGRAM_INDEX_TESS_EVALUATION],
				change_array + GLD_EP_BITFIELD_OFFSET,
				GLD_PROGRAM_STATE_BIT);
		}
	}
	#endif
	#endif /* GLR_HAS_TESSELLATION_SHADER */

	#if GLR_HAS_GEOMETRY_SHADER
	/* Handle Geometry Shaders */
	if(change_array[3] & (
		#if GLR_PIPEPROG_GEOM_CTX_DIRTY_MASK & GLD_PROGRAM_STATE_BIT
		GLD_STATE3_GEOMETRY_PROGRAM_STATE |
		#endif
		GLD_STATE3_PROGRAM_OBJECT_BIND |
		GLD_STATE3_GEOMETRY_PROGRAM_BIND |
		GLD_STATE3_PROCESSING_MODE))
	{
		#if GLR_PIPEPROG_NEEDS_DISABLE_NOTIFIER
		old_code = ctx->disable_code;
		#endif /* GLR_PIPEPROG_NEEDS_DISABLE_NOTIFIER */
		ctx->disable_code &= ~(
			GPUS_FALLBACK_VERTEX_FOR_GEOM_PROG |
			GPUS_FALLBACK_VERTEX_FOR_GEOM_BUF_REQ |
			GPUS_FALLBACK_FRAGMENT_FOR_VERTEX_FALLBACK);

		program_obj = ctx->plugin_state->programs[GLD_PROGRAM_INDEX_GEOMETRY];

		#if GLR_ASSERT
		if(program_obj)
		{
			gpusAssert(ctx->plugin_state->programs[GLD_PROGRAM_INDEX_VERTEX]); // if there is a geometry shader there must be a vertex shader
			gpusAssert(ctx->plugin_state->programs[GLD_PROGRAM_INDEX_VERTEX]->program_data->valid_shaders & GLD_SHADER_BIT_VERTEX); // the vertex shader must have a valid vertex shader
			gpusAssert(ctx->plugin_state->programs[GLD_PROGRAM_INDEX_GEOMETRY]->program_data->valid_shaders & GLD_SHADER_BIT_GEOMETRY); // the geometry shader must have a valid geometry shader
		}
		#endif

		/* Check if the vertex stage is disabled. */
		if(ctx->disable_code & (
			GPUS_FALLBACK_VERTEX_EXTERNAL_MASK |
			GPUS_FALLBACK_FRAGMENT_EXTERNAL_MASK))
		{
			program_obj = NULL;
			pipeprog_obj = NULL;
		}
		/* Check for GLSL */
		else if(program_obj && (program_obj->program_data->valid_shaders & GLD_SHADER_BIT_GEOMETRY))
		{
			/* Trying to use GLSL. */
			/* If a transform feedback varying present but the driver does not implement
			** transform feedback, fail to load vertex shader */
			if(glsl_disabled)
			{
				ctx->disable_code |= GPUS_FALLBACK_VERTEX_FOR_GEOM_BUF_REQ;
				program_obj = NULL;
				pipeprog_obj = NULL;
			}
			else
			{
				pipeprog_obj = program_obj->plugin_program_data->pipeline_shaders[GLD_SHADER_INDEX_GEOMETRY];
				gpusAssert(pipeprog_obj);
				gpusAssert(pipeprog_obj->program_data->data || pipeprog_obj->program_data->new_ir_data);
			}
		}
		/* No geometry shader is enabled */
		else
		{
			program_obj = NULL;
			pipeprog_obj = NULL;
		}

		#if GLR_PIPEPROG_NEEDS_DISABLE_NOTIFIER
		if(ctx->disable_code != old_code)
			glrUpdateForDisableCodeChange(ctx, old_code);
		#endif // GLR_PIPEPROG_NEEDS_DISABLE_NOTIFIER

		glrLoadCurrentGeometryProgram(ctx, program_obj, pipeprog_obj, change_array);
	}
	#if (GLR_PIPEPROG_GEOM_BIND_DIRTY_MASK & GLD_PROGRAM_STATE_BIT) && \
		!(GLR_PIPEPROG_GEOM_CTX_DIRTY_MASK & GLD_PROGRAM_STATE_BIT)
	else if(change_array[3] & GLD_STATE3_GEOMETRY_PROGRAM_STATE)
	{
		if(ctx->cur_pipeline_program[GLD_PROGRAM_INDEX_GEOMETRY])
		{
			program_obj = ctx->plugin_state->programs[GLD_PROGRAM_INDEX_GEOMETRY];
			gpusAssert(program_obj &&
				(program_obj->program_data->valid_shaders & GLD_SHADER_BIT_GEOMETRY));

			glrBindSysGeometryProgram(ctx,
				program_obj,
				ctx->cur_pipeline_program[GLD_PROGRAM_INDEX_GEOMETRY],
				ctx->prog_vend[GLD_PROGRAM_INDEX_GEOMETRY],
				change_array + GLD_GP_BITFIELD_OFFSET,
				GLD_PROGRAM_STATE_BIT);
		}
	}
	#endif
	#endif /* GLR_HAS_GEOMETRY_SHADER */

	#if GLR_HAS_VERTEX_SHADER
	// The set of bits tested here is the union of the vertex, tessellation, and geometry bits tested above.
	if(change_array[3] & (
		#if GLR_PIPEPROG_VERT_CTX_DIRTY_MASK & GLD_PROGRAM_STATE_BIT
		GLD_STATE3_VERTEX_PROGRAM_STATE |
		#endif
		#if GLR_HAS_TESSELLATION_SHADER
		#if GLR_PIPEPROG_CTRL_CTX_DIRTY_MASK & GLD_PROGRAM_STATE_BIT
		GLD_STATE3_TESS_CONTROL_PROGRAM_STATE |
		#endif
		#if GLR_PIPEPROG_EVAL_CTX_DIRTY_MASK & GLD_PROGRAM_STATE_BIT
		GLD_STATE3_TESS_EVALUATION_PROGRAM_STATE |
		#endif
		#endif
		#if GLR_HAS_GEOMETRY_SHADER
		#if GLR_PIPEPROG_GEOM_CTX_DIRTY_MASK & GLD_PROGRAM_STATE_BIT
		GLD_STATE3_GEOMETRY_PROGRAM_STATE |
		#endif
		#endif
		GLD_STATE3_PROGRAM_OBJECT_BIND |
		GLD_STATE3_VERTEX_PROGRAM_BIND |
		GLD_STATE3_TESS_CONTROL_PROGRAM_BIND |
		GLD_STATE3_TESS_EVALUATION_PROGRAM_BIND |
		GLD_STATE3_GEOMETRY_PROGRAM_BIND |
		GLD_STATE3_PROCESSING_MODE))
	{
		if(ctx->disable_code & GPUS_FALLBACK_VERTEX_MASK)
		{
			pipeprog_obj = ctx->plugin_state->fallback_program[GLD_FALLBACK_INDEX_VERTEX_VERTFALL];
			
			uint32_t program_index;
			program_index = GLD_PROGRAM_INDEX_VERTEX;
			if(ctx->cur_pipeline_program[program_index] != pipeprog_obj)
			{
				GLRVendBoundPipeProg vendpipeprog;
				GLRCookie *program_cookie;

				program_obj  = NULL;

				vendpipeprog = glrUpdateVertexProgramInline(ctx, program_obj, pipeprog_obj,
					&program_cookie, 0U);
				if(!vendpipeprog)
				{
					pipeprog_obj = NULL;
					ctx->disable_code |= GPUS_FALLBACK_FRAGMENT_FOR_VERTEX_FALLBACK;
					change_array[3] |=
						GLD_STATE3_FRAGMENT_PROGRAM_BIND | GLD_STATE3_PROGRAM_OBJECT_BIND;
				}
				ctx->ctx_prog_cookie[program_index] = *program_cookie;

				glrBindSysProgram[program_index](ctx,
					program_obj,
					pipeprog_obj,
					vendpipeprog,
					change_array + bitfield_offsets[program_index],
					bind_dirty_masks[program_index]);

				if(vendpipeprog)
				{
					#if GLR_PIPEPROG_HAS_CTX_BUILD
					glrRetainVendCtxPipeProg(ctx, vendpipeprog);
					#else
					glrRetainVendShrPipeProg(ctx->shared, vendpipeprog);
					#endif
				}

				ctx->prog_vend[program_index] = vendpipeprog;
				ctx->cur_pipeline_program[program_index] = pipeprog_obj;
			}

			#if GLR_HAS_TESSELLATION_SHADER
			program_index = GLD_PROGRAM_INDEX_TESS_CONTROL;
			if(ctx->cur_pipeline_program[program_index])
			{
				GLRVendBoundPipeProg vendpipeprog;
				gpusAssert(ctx->prog_vend[program_index]);
				program_obj  = NULL;

				pipeprog_obj = NULL;
				vendpipeprog = NULL;
				gpumSetCookieNoProgram(&ctx->ctx_prog_cookie[program_index]);

				glrBindSysProgram[program_index](ctx,
					program_obj,
					pipeprog_obj,
					vendpipeprog,
					NULL,
					bind_dirty_masks[program_index]);

				#if GLR_PIPEPROG_HAS_CTX_BUILD
				glrReleaseVendCtxPipeProg(ctx, ctx->prog_vend[program_index]);
				#else
				glrReleaseVendShrPipeProg(ctx->shared, ctx->prog_vend[program_index]);
				#endif

				ctx->prog_vend[program_index] = vendpipeprog;
				ctx->cur_pipeline_program[program_index] = pipeprog_obj;
			}

			program_index = GLD_PROGRAM_INDEX_TESS_EVALUATION;
			if(ctx->cur_pipeline_program[program_index])
			{
				GLRVendBoundPipeProg vendpipeprog;
				gpusAssert(ctx->prog_vend[program_index]);
				program_obj  = NULL;

				pipeprog_obj = NULL;
				vendpipeprog = NULL;
				gpumSetCookieNoProgram(&ctx->ctx_prog_cookie[program_index]);

				glrBindSysProgram[program_index](ctx,
					program_obj,
					pipeprog_obj,
					vendpipeprog,
					NULL,
					bind_dirty_masks[program_index]);

				#if GLR_PIPEPROG_HAS_CTX_BUILD
				glrReleaseVendCtxPipeProg(ctx, ctx->prog_vend[program_index]);
				#else
				glrReleaseVendShrPipeProg(ctx->shared, ctx->prog_vend[program_index]);
				#endif

				ctx->prog_vend[program_index] = vendpipeprog;
				ctx->cur_pipeline_program[program_index] = pipeprog_obj;
			}
			#endif /* GLR_HAS_TESSELLATION_SHADER */

			#if GLR_HAS_GEOMETRY_SHADER
			program_index = GLD_PROGRAM_INDEX_GEOMETRY;
			if(ctx->cur_pipeline_program[program_index])
			{
				GLRVendBoundPipeProg vendpipeprog;
				gpusAssert(ctx->prog_vend[program_index]);
				program_obj  = NULL;

				pipeprog_obj = NULL;
				vendpipeprog = NULL;
				gpumSetCookieNoProgram(&ctx->ctx_prog_cookie[program_index]);

				glrBindSysProgram[program_index](ctx,
					program_obj,
					pipeprog_obj,
					vendpipeprog,
					NULL,
					bind_dirty_masks[program_index]);

				#if GLR_PIPEPROG_HAS_CTX_BUILD
				glrReleaseVendCtxPipeProg(ctx, ctx->prog_vend[program_index]);
				#else
				glrReleaseVendShrPipeProg(ctx->shared, ctx->prog_vend[program_index]);
				#endif

				ctx->prog_vend[program_index] = vendpipeprog;
				ctx->cur_pipeline_program[program_index] = pipeprog_obj;
			}
			#endif /* GLR_HAS_GEOMETRY_SHADER */
		}
	}
	#endif // GLR_HAS_VERTEX_SHADER

	/* Handle Fragment Programs/Shaders */
	if(change_array[3] & (
		#if GLR_PIPEPROG_FRAG_CTX_DIRTY_MASK & GLD_PROGRAM_STATE_BIT
		GLD_STATE3_FRAGMENT_PROGRAM_STATE |
		#endif
		GLD_STATE3_PROGRAM_OBJECT_BIND |
		GLD_STATE3_FRAGMENT_PROGRAM_BIND |
		GLD_STATE3_PROCESSING_MODE))
	{
		#if GLR_PIPEPROG_NEEDS_DISABLE_NOTIFIER
		old_code = ctx->disable_code;
		#endif /* GLR_PIPEPROG_NEEDS_DISABLE_NOTIFIER */
		ctx->disable_code &= ~(
			GPUS_FALLBACK_FRAGMENT_FOR_FRAG_PROG |
			GPUS_FALLBACK_FRAGMENT_FOR_FRAG_BUF_REQ);

		program_obj = ctx->plugin_state->programs[GLD_PROGRAM_INDEX_FRAGMENT];

		/* Check if the fragment stage is disabled. */
		if(ctx->disable_code & (
			GPUS_FALLBACK_FRAGMENT_EXTERNAL_MASK |
			GPUS_FALLBACK_FRAGMENT_FOR_VERTEX_FALLBACK))
		{
			program_obj = NULL;
			pipeprog_obj = NULL;
		}
		/* Check for GLSL */
		else if(program_obj && (program_obj->program_data->valid_shaders & GLD_SHADER_BIT_FRAGMENT))
		{
			/* Trying to use GLSL. */
			if(glsl_disabled)
			{
				ctx->disable_code |= GPUS_FALLBACK_FRAGMENT_FOR_FRAG_BUF_REQ;
				program_obj = NULL;
				pipeprog_obj = NULL;
			}
			else
			{
				pipeprog_obj = program_obj->plugin_program_data->pipeline_shaders[GLD_SHADER_INDEX_FRAGMENT];
				gpusAssert(pipeprog_obj);
				gpusAssert(pipeprog_obj->program_data->data || pipeprog_obj->program_data->new_ir_data);
			}
		}
		/* Check for ARB */
		else
		{
			program_obj = NULL;
			pipeprog_obj = ctx->plugin_state->pipeline_program[GLD_PROGRAM_INDEX_FRAGMENT];
			gpusAssert(pipeprog_obj);
			if(!pipeprog_obj->program_data->data && !pipeprog_obj->program_data->new_ir_data)
			{
				// We can get here in glBeginTransformFeedback with RASTERIZER_DISCARD not yet set
				// and it should not be an error. So if we won't assert here if transform
				// feedback is enabled
				gpusAssert(ctx->state->transform_feedback.primitive_mode != GLD_TRANSFORM_FEEDBACK_PRIMITIVE_MODE_NONE ||
					ctx->state->transform_feedback.rasterizer_discard);
				// This special handling is needed only for rasterizer_discard.  We could have an
				// incomplete fragment shader but still need a complete render pipeline for
				// transform_feedback.
				pipeprog_obj =
					ctx->plugin_state->fallback_program[GLD_FALLBACK_INDEX_FRAGMENT_FRAGFALL];
				gpusAssert(pipeprog_obj);
				gpusAssert(pipeprog_obj->program_data->data || pipeprog_obj->program_data->new_ir_data);
			}
		}

		#if GLR_PIPEPROG_NEEDS_DISABLE_NOTIFIER
		if(ctx->disable_code != old_code)
			glrUpdateForDisableCodeChange(ctx, old_code);
		#endif // GLR_PIPEPROG_NEEDS_DISABLE_NOTIFIER

		glrLoadCurrentFragmentProgram(ctx, program_obj, pipeprog_obj, change_array);

		#if GLR_HAS_VERTEX_SHADER || GLR_HAS_GEOMETRY_SHADER
		// If we have fallen back while processing the fragment stage, unbind the vertex and
		// geometry stages.
		#if GLR_ASSERT
		if(ctx->disable_code & GPUS_FALLBACK_VERTEX_MASK)
		{
			#if GLR_HAS_GEOMETRY_SHADER
			gpusAssert(!ctx->cur_pipeline_program[GLD_PROGRAM_INDEX_GEOMETRY]);
			gpusAssert(!ctx->prog_vend[GLD_PROGRAM_INDEX_GEOMETRY]);
			gpusAssert(gpumIsCookieNoProgram(&(ctx->ctx_prog_cookie[GLD_PROGRAM_INDEX_GEOMETRY])));
			#endif
		}
		#endif
		#endif

		if(ctx->disable_code & GPUS_FALLBACK_FRAGMENT_MASK)
		{
			uint32_t program_index;
			#if GLR_HAS_VERTEX_SHADER
			pipeprog_obj = ctx->plugin_state->fallback_program[GLD_FALLBACK_INDEX_VERTEX_FRAGFALL];
			

			program_index = GLD_PROGRAM_INDEX_VERTEX;
			if(ctx->cur_pipeline_program[program_index] != pipeprog_obj)
			{
				GLRVendBoundPipeProg vendpipeprog = NULL;
				GLRCookie *program_cookie;
				program_obj  = NULL;
				
				if (pipeprog_obj)
				{
					vendpipeprog = glrUpdateVertexProgramInline(ctx, program_obj, pipeprog_obj,
						&program_cookie, 0U);
					
					if(vendpipeprog)
					{
						ctx->ctx_prog_cookie[program_index] = *program_cookie;

						glrBindSysProgram[program_index](ctx,
							program_obj,
							pipeprog_obj,
							vendpipeprog,
							change_array + bitfield_offsets[program_index],
							bind_dirty_masks[program_index]);

						#if GLR_PIPEPROG_HAS_CTX_BUILD
						glrRetainVendCtxPipeProg(ctx, vendpipeprog);
						#else
						glrRetainVendShrPipeProg(ctx->shared, vendpipeprog);
						#endif
					}
				}
				if(ctx->prog_vend[program_index])
				{
					#if GLR_PIPEPROG_HAS_CTX_BUILD
					glrReleaseVendCtxPipeProg(ctx, ctx->prog_vend[program_index]);
					#else
					glrReleaseVendShrPipeProg(ctx->shared, ctx->prog_vend[program_index]);
					#endif
				}

				ctx->prog_vend[program_index] = vendpipeprog;
				ctx->cur_pipeline_program[program_index] = pipeprog_obj;
			}

			#if GLR_HAS_TESSELLATION_SHADER
			program_index = GLD_PROGRAM_INDEX_TESS_CONTROL;
			if(ctx->cur_pipeline_program[program_index])
			{
				GLRVendBoundPipeProg vendpipeprog;
				gpusAssert(ctx->prog_vend[program_index]);
				program_obj  = NULL;

				pipeprog_obj = NULL;
				vendpipeprog = NULL;
				gpumSetCookieNoProgram(&ctx->ctx_prog_cookie[program_index]);

				glrBindSysProgram[program_index](ctx,
					program_obj,
					pipeprog_obj,
					vendpipeprog,
					NULL,
					bind_dirty_masks[program_index]);

				#if GLR_PIPEPROG_HAS_CTX_BUILD
				glrReleaseVendCtxPipeProg(ctx, ctx->prog_vend[program_index]);
				#else
				glrReleaseVendShrPipeProg(ctx->shared, ctx->prog_vend[program_index]);
				#endif

				ctx->prog_vend[program_index] = vendpipeprog;
				ctx->cur_pipeline_program[program_index] = pipeprog_obj;
			}

			program_index = GLD_PROGRAM_INDEX_TESS_EVALUATION;
			if(ctx->cur_pipeline_program[program_index])
			{
				GLRVendBoundPipeProg vendpipeprog;
				gpusAssert(ctx->prog_vend[program_index]);
				program_obj  = NULL;

				pipeprog_obj = NULL;
				vendpipeprog = NULL;
				gpumSetCookieNoProgram(&ctx->ctx_prog_cookie[program_index]);

				glrBindSysProgram[program_index](ctx,
					program_obj,
					pipeprog_obj,
					vendpipeprog,
					NULL,
					bind_dirty_masks[program_index]);

				#if GLR_PIPEPROG_HAS_CTX_BUILD
				glrReleaseVendCtxPipeProg(ctx, ctx->prog_vend[program_index]);
				#else
				glrReleaseVendShrPipeProg(ctx->shared, ctx->prog_vend[program_index]);
				#endif

				ctx->prog_vend[program_index] = vendpipeprog;
				ctx->cur_pipeline_program[program_index] = pipeprog_obj;
			}
			#endif /* GLR_HAS_TESSELLATION_SHADER */

			#if GLR_HAS_GEOMETRY_SHADER
			program_index = GLD_PROGRAM_INDEX_GEOMETRY;
			if(ctx->cur_pipeline_program[program_index])
			{
				GLRVendBoundPipeProg vendpipeprog;
				gpusAssert(ctx->prog_vend[program_index]);
				program_obj  = NULL;

				pipeprog_obj = NULL;
				vendpipeprog = NULL;
				gpumSetCookieNoProgram(&ctx->ctx_prog_cookie[program_index]);

				glrBindSysProgram[program_index](ctx,
					program_obj,
					pipeprog_obj,
					vendpipeprog,
					NULL,
					bind_dirty_masks[program_index]);

				#if GLR_PIPEPROG_HAS_CTX_BUILD
				glrReleaseVendCtxPipeProg(ctx, ctx->prog_vend[program_index]);
				#else
				glrReleaseVendShrPipeProg(ctx->shared, ctx->prog_vend[program_index]);
				#endif

				ctx->prog_vend[program_index] = vendpipeprog;
				ctx->cur_pipeline_program[program_index] = pipeprog_obj;
			}
			#endif /* GLR_HAS_GEOMETRY_SHADER */
			#endif /* GLR_HAS_VERTEX_SHADER */

			pipeprog_obj = ctx->plugin_state->fallback_program[GLD_FALLBACK_INDEX_FRAGMENT_FRAGFALL];
			gpusAssert(pipeprog_obj);
			program_index = GLD_PROGRAM_INDEX_FRAGMENT;
			if(ctx->cur_pipeline_program[program_index] != pipeprog_obj)
			{
				GLRVendBoundPipeProg vendpipeprog;
				GLRCookie *program_cookie;
				program_obj = NULL;

				vendpipeprog = glrUpdateFragmentProgramInline(ctx, program_obj, pipeprog_obj,
					&program_cookie, 0U);
				ctx->ctx_prog_cookie[program_index] = *program_cookie;

				if (vendpipeprog)
				{
					glrBindSysProgram[program_index](ctx,
						program_obj,
						pipeprog_obj,
						vendpipeprog,
						change_array + bitfield_offsets[program_index],
						bind_dirty_masks[program_index]);

					#if GLR_PIPEPROG_HAS_CTX_BUILD
					glrRetainVendCtxPipeProg(ctx, vendpipeprog);
					#else
					glrRetainVendShrPipeProg(ctx->shared, vendpipeprog);
					#endif
				}
				if(ctx->prog_vend[program_index])
				{
					#if GLR_PIPEPROG_HAS_CTX_BUILD
					glrReleaseVendCtxPipeProg(ctx, ctx->prog_vend[program_index]);
					#else
					glrReleaseVendShrPipeProg(ctx->shared, ctx->prog_vend[program_index]);
					#endif
				}

				ctx->prog_vend[program_index] = vendpipeprog;
				ctx->cur_pipeline_program[program_index] = pipeprog_obj;
			}
		}
	}
	#if (GLR_PIPEPROG_FRAG_BIND_DIRTY_MASK & GLD_PROGRAM_STATE_BIT) && \
		!(GLR_PIPEPROG_FRAG_CTX_DIRTY_MASK & GLD_PROGRAM_STATE_BIT)
	else if(change_array[3] & GLD_STATE3_FRAGMENT_PROGRAM_STATE)
	{
		if(ctx->cur_pipeline_program[GLD_PROGRAM_INDEX_FRAGMENT])
		{
			program_obj = ctx->plugin_state->programs[GLD_PROGRAM_INDEX_FRAGMENT];
			if(program_obj && !(program_obj->program_data->valid_shaders & GLD_SHADER_BIT_FRAGMENT))
				program_obj = NULL;

			glrBindSysFragmentProgram(ctx,
				program_obj,
				ctx->cur_pipeline_program[GLD_PROGRAM_INDEX_FRAGMENT],
				ctx->prog_vend[GLD_PROGRAM_INDEX_FRAGMENT],
				change_array + GLD_FP_BITFIELD_OFFSET,
				GLD_PROGRAM_STATE_BIT);
		}
	}
	#endif

	if(change_array[3] & (
		GLD_STATE3_PROGRAM_OBJECT_BIND |
		GLD_STATE3_PROCESSING_MODE))
	{
		int shaderTarget;
		for(shaderTarget = 0; shaderTarget < GLD_NUM_PROGRAM_TARGETS; shaderTarget++)
		{
			program_obj = ctx->plugin_state->programs[shaderTarget];
			if(program_obj)
			{
				program_obj->plugin_program_data->program_dirty_bits = 0U;

				/* clean all uniforms */
				memset(program_obj->plugin_program_data->program_uniforms_dirty_bits, 0x00, ((program_obj->program_data->program_uniforms_number + 0x1F) >> 5) * sizeof(bitfield32_t));
				program_obj->plugin_program_data->program_uniforms_min_dirty = program_obj->program_data->program_uniforms_number;
				program_obj->plugin_program_data->program_uniforms_max_dirty = -1;
			}
		}
	}
}

