
#import "MainMenuSettings.js"

/**
  * Function: mainmenusettings_highlightItem()
  *     Go to Settings>Main Menu and highlight item. The itemName is assumed to be in
  *     US English.
  *
  * \param itemName (string)    - Item on main menu to highlight
  * \param processName (string) - (optional) Process to take information
  * \param infoFlags (string)   - (optional) Flag for information to take
  *
  * \return 1 if success, otherwise 0
  **/
function mainmenusettings_goToMainMenuSettings() {
    if (this.argv.length < 1) {
        this.testError = "Item to highlight is required";
        return this.testResult = 0;
    }
    var itemName = this.argv[0];
    var processName = undefined;
    var infoFlags = undefined;
    if (this.argv.length >= 2) {
        processName = this.argv[1];
        if (this.argv.length >= 3) {
            infoFlags = this.argv[2];
        }
        else {
            infoFlags = "";
        }
    }

    if ( ! mainmenusettings.goToAndHighlight(mainmenusettings.localizedStringFromEnglish(itemName))) {
        this.testError = "Failed to highlight item";
        this.testResult = 0;
    }
    else {
        this.testResult = 1;
    }

    if (processName) {
        // wait 5 seconds before taking heap
        atvaTarget.delay(5);
        var output = new ATVAOutput("/var/root/heaps");
        output.addInfomationToCollect(processName, infoFlags);
        output.collectProcessInfomation("On Settings>Main Menu>" + itemName);
    }
    return this.testResult;
}

/**
  * Function: mainmenusettings_verifyMenuContents()
  *     Go to Settings>Main Menu and verify expected menu item.
  *
  * \return 1 if success, otherwise 0
  **/
function mainmenusettings_verifyMenuContents() {
    if ( ! mainmenusettings.goTo()) {
        this.testError = "Fail to get to Settings>Main Menu";
        return this.testResult = 0;
    }

    if ( ! mainmenusettings.appMenu().verifyMenuContents(mainmenusettings.menuItems()) ) {
        this.testError = "Menu contents verification failed";
        return this.testResult = 0;
    }
    else {
        return this.testResult = 1;
    }
}

/**
  * Function: mainmenusettings_verifyMenuNavigation()
  *     Go to Settings>Main Menu and verify basic navigation.
  *
  * \return 1 if success, otherwise 0
  **/
function mainmenusettings_verifyMenuNavigation() {
    if ( ! mainmenusettings.goTo()) {
        this.testError = "Fail to get to Settings";
        return this.testResult = 0;
    }

    if ( ! mainmenusettings.appMenu().verifyMenuNavigation(mainmenusettings.menuItems(), mainmenusettings.menuItemsNavigation()) ) {
        this.testError = "Menu navigation verification failed";
        return this.testResult = 0;
    }
    else {
        return this.testResult = 1;
    }
}

/**
  * Function: mainmenusettings_showItem()
  *     Item must be hidden before calling this test.
  *
  *     Goes to Main Menu and verifies that the item is not shown.
  *     Goes to Settings>Main Menu and sets the item to Show.
  *     Goes back to Main Menu and verifies that item is now visible.
  *
  * \param itemName (string) - Item on main menu to show
  *
  * \return 1 if success, otherwise 0
  **/
function mainmenusettings_showItem() {
    if (this.argv.length < 1) {
        this.testError = "Item required";
        return this.testResult = 0;
    }
    var itemName = this.argv[0];

    if ( ! mainmenu.goTo()) {
        this.testError = "Failed get to Main Menu";
        return(this.testResult = 0);
    }
    if (mainmenu.appMenu().findCategoryElementWithName(mainmenu.localizedStringFromEnglish(itemName))) {
        this.testError = "Failed, item is already shown \"" + itemName + "\"";
        return(this.testResult = 0);
    }
    if ( ! mainmenusettings.goToAndSetItem(mainmenusettings.localizedStringFromEnglish(itemName), true)) {
        this.testError = "Failed to set item to show \"" + itemName + "\"";
        return(this.testResult = 0);
    }
    // need to reset grid because it has changed
    mainmenu.appMenu().resetGrid();
    if ( ! mainmenu.goTo()) {
        this.testError = "Failed get to return to Main Menu";
        return(this.testResult = 0);
    }
    if ( ! mainmenu.appMenu().findCategoryElementWithName(mainmenu.localizedStringFromEnglish(itemName))) {
        this.testError = "Failed, item is not shown \"" + itemName + "\"";
        return(this.testResult = 0);
    }

    return(this.testResult = 1);
}

/**
  * Function: mainmenusettings_hideItem()
  *     Item must be shown before calling this test.
  *
  *     Goes to Main Menu and verifies that the item is shown.
  *     Goes to Settings>Main Menu and sets the item to Hide.
  *     Goes back to Main Menu and verifies that item is now not visible.
  *
  * \param itemName (string) - Item on main menu to hide
  *
  * \return 1 if success, otherwise 0
  **/
function mainmenusettings_hideItem() {
    if (this.argv.length < 1) {
        this.testError = "Item required";
        return this.testResult = 0;
    }
    var itemName = this.argv[0];

    if ( ! mainmenu.goTo()) {
        this.testError = "Failed get to Main Menu";
        return(this.testResult = 0);
    }
    if ( ! mainmenu.appMenu().findCategoryElementWithName(mainmenu.localizedStringFromEnglish(itemName))) {
        this.testError = "Failed, item is already hidden \"" + itemName + "\"";
        return(this.testResult = 0);
    }
    if ( ! mainmenusettings.goToAndSetItem(mainmenusettings.localizedStringFromEnglish(itemName), false)) {
        this.testError = "Failed to set item to hide \"" + itemName + "\"";
        return(this.testResult = 0);
    }
    // need to reset grid because it has changed
    mainmenu.appMenu().resetGrid();
    if ( ! mainmenu.goTo()) {
        this.testError = "Failed get to return to Main Menu";
        return(this.testResult = 0);
    }
    if (mainmenu.appMenu().findCategoryElementWithName(mainmenu.localizedStringFromEnglish(itemName))) {
        this.testError = "Failed, item is not hidden \"" + itemName + "\"";
        return(this.testResult = 0);
    }

    return(this.testResult = 1);
}
