
#import "../Settings.js"

/**
 *  Object: mainmenusettings
 *      ATVAApp object for the Settings>Main Menu. Typical actions on the Settings>Main Menu are
 *          1. Go to and set item to Show
 *          2. Go to and set item to Hide
 **/

if (typeof mainmenusettings == 'undefined') {
    mainmenusettings = new ATVAApp("Settings");
}

if (typeof mainmenusettings != 'undefined') {
    mainmenusettings._objectName = "mainmenusettings";

    // Item localization keys
    mainmenusettings.kMainMenuKey = "SETTINGSMainMenuName";
    mainmenusettings.kParentalControlsShowKey = "RUISettingParentalControlsShow";
    mainmenusettings.kParentalControlsHideKey = "RUISettingParentalControlsHide";
    mainmenusettings.kParentalControlsRestrictedKey = "RUISettingParentalControlsRestricted";

    mainmenusettings.kMainMenuEnglishString = "Main Menu";
    mainmenusettings.kShowEnglishString = "Show";
    mainmenusettings.kHideEnglishString = "Hide";
    mainmenusettings.kRestrictedEnglishString = "Restricted";

    /**
      * Function: mainmenusettings.menuItems()
      *     Returns an array of strings for the items in the menu.
      *
      *  \return array of strings (items in menu)
      **/
    mainmenusettings._menuItems = undefined;
    mainmenusettings.menuItems = function menuItems() {
        if ( ! this._menuItems) {
            try {
                var fd = new File("/usr/local/etc/scripterUIA/appletv/settings/mainmenu/MainMenuSettingsInfo.json", "r");
                var info = JSON.parse(fd.read());

                this._menuItems = info['menuItems'];
            }
            catch (e) {
                UIALogger.logError("Cannot read info to \"/usr/local/etc/scripterUIA/appletv/settings/mainmenu/MainMenuSettingsInfo.json\"");
            }
        }
        return(this._menuItems);
    }

    /**
      * Function: mainmenusettings.menuItemsDefaultValues()
      *     Returns an dictionary of the menu items with there default state (localized).
      *
      *  \return array of strings (items in menu)
      **/
    mainmenusettings._menuItemsDefaultValues = undefined;
    mainmenusettings.menuItemsDefaultValues = function menuItemsDefaultValues() {
        if ( ! this._menuItemsDefaultValues) {
            try {
                var fd = new File("/usr/local/etc/scripterUIA/appletv/settings/mainmenu/MainMenuSettingsInfo.json", "r");
                var info = JSON.parse(fd.read());
                var keys = Object.keys(info['menuItemsDefaultValues']);

                this._menuItemsDefaultValues = {};
                for (var i = 0; i < keys.length; ++i) {
                    this._menuItemsDefaultValues[keys[i]] = this.localizedStringFromEnglish(info['menuItemsDefaultValues'][keys[i]]);
                }
            }
            catch (e) {
                UIALogger.logError("Cannot read info to \"/usr/local/etc/scripterUIA/appletv/settings/mainmenu/MainMenuSettingsInfo.json\"");
            }
        }
        return(this._menuItemsDefaultValues);
    }

    /**
      * Function: settings.menuItemsNavigation()
      *     Returns an dictionary of object for the items in the menu. The data in the object
      *     describes all the possible navigation for each menu item. Notice that the the data
      *     is loaded from a JSON file and menu items are listed with the English names.
      *
      *  \return dictionary of objects
      **/
    mainmenusettings._menuItemsNavigation = undefined;
    mainmenusettings.menuItemsNavigation = function menuItemsNavigation() {
        if ( ! this._menuItemsNavigation) {
            try {
                var fd = new File("/usr/local/etc/scripterUIA/appletv/settings/mainmenu/MainMenuSettingsInfo.json", "r");
                var info = JSON.parse(fd.read());
                var keys = Object.keys(info['navigation']);

                this._menuItemsNavigation = {};
                for (var i = 0; i < keys.length; ++i) {
                    this._menuItemsNavigation[this.localizedStringFromEnglish(keys[i])] = info['navigation'][keys[i]];
                }
            }
            catch (e) {
                UIALogger.logError("Cannot read info to \"/usr/local/etc/scripterUIA/appletv/settings/mainmenu/MainMenuSettingsInfo.json\"");
            }
        }
        return(this._menuItemsNavigation);
    }

    /**
      * Function: mainmenusettings.setLocalizedStrings()
      *     Sets up the localization strings for this area. This is overriding
      *     ATVAApp.setLocalizedStrings.
      *
      *     Normally, this function is not run directly, but called the first time
      *     either mainmenusettings.localizedString() or mainmenusettings.localizedStringFromEnglish()
      *     are called.
      *
      *  \return localized string
      **/
    mainmenusettings.setLocalizedStrings = function setLocalizedStrings() {
        this._localizedStrings = {};
        this._localizedStrings[this.kMainMenuKey] = UIATarget.localTarget().localizedStringForKeyValueTableBundlePath(this.kMainMenuKey, "", "SettingsLocalizable", "/Applications/AppleTV.app");
        this._localizedStrings[this.kParentalControlsShowKey] = UIATarget.localTarget().localizedStringForKeyValueTableBundlePath(this.kParentalControlsShowKey, "", "SettingsLocalizable", "/Applications/AppleTV.app");
        this._localizedStrings[this.kParentalControlsHideKey] = UIATarget.localTarget().localizedStringForKeyValueTableBundlePath(this.kParentalControlsHideKey, "", "SettingsLocalizable", "/Applications/AppleTV.app");
        this._localizedStrings[this.kParentalControlsRestrictedKey] = UIATarget.localTarget().localizedStringForKeyValueTableBundlePath(this.kParentalControlsRestrictedKey, "", "SettingsLocalizable", "/Applications/AppleTV.app");

        this._englishToKey = {};
        this._englishToKey[this.kMainMenuEnglishString] = this.kMainMenuKey;
        this._englishToKey[this.kShowEnglishString] = this.kParentalControlsShowKey;
        this._englishToKey[this.kHideEnglishString] = this.kParentalControlsHideKey;
        this._englishToKey[this.kRestrictedEnglishString] = this.kParentalControlsRestrictedKey;
    }

    // Title of ATVAApp is not localized yet, localize
    mainmenusettings.setTitle(UIATarget.localTarget().localizedStringForKeyValueTableBundlePath(mainmenusettings.kMainMenuKey, "", "SettingsLocalizable", "/Applications/AppleTV.app"));

    /**
      * Function: mainmenusettings.appMenu()
      *     If the ATVAPageList object has not been created, creates it and then
      *     returns it. This menu is always short enough and does not scroll.
      *
      *  \return ATVAPageList object
      **/
    mainmenusettings.appMenu = function appMenu() {
        if ( ! this._appMenu) {
            this._appMenu = new ATVAPageList(this.localizedString(this.kMainMenuKey));
        }
        return(this._appMenu);
    }

    /**
      * Function: mainmenusettings.fromMenu(interval, timeout)
      *     Gets the _appMenu and waits for it to be displayed, checking every
      *     interval seconds until displayed or timeout is reached.
      *
      *     Throws exception if waiting for Settings to be display timeout.
      *
      * \param interval (number) - Seconds between checking screen for main menu.
      *                            Defaults to 5.
      * \param timeout (number)  - Seconds between checking screen for main menu.
      *                            Defaults to 30.
      *
      * \return ATVAPageList object on success, undefined on failure
      **/
    mainmenusettings.fromMenu = function fromMenu(interval, timeout) {
        var oldInterval = undefined;
        var oldTimeout = undefined;

        if (typeof interval != 'undefined') {
            oldInterval = this.appMenu()._interval;
            this.appMenu()._interval = interval;
        }
        if (typeof timeout != 'undefined') {
            oldTimeout = this.appMenu()._timeout;
            this.appMenu()._timeout = timeout;
        }

        if ( ! this.appMenu().waitForPage()) {
            throw new Error("Failed waiting for \"" + this._title + "\"");
        }

        // reset interval and timeout
        if (typeof oldInterval != 'undefined') {
            this.appMenu()._interval = oldInterval;
        }
        if (typeof oldTimeout != 'undefined') {
            this.appMenu()._timeout = oldTimeout;
        }

        return(this.appMenu());
    }

    /**
      * Function: mainmenusettings.fromMenuAndHighlight(name, count, interval, timeout)
      *     Gets the _appMenu and waits for it to be displayed, checking every
      *     interval seconds until displayed or timeout is reached. Then moves
      *     to and highlight to the count'th instance of name.
      *
      * \param name (string)     - The localized name for the menu item to move to.
      * \param count (number)    - When there is more than one item with the same
      *                            name, then a count is required to select past the
      *                            first one. Notice count == undefined (assumes there
      *                            can only be 1) which is different than count == 1.
      *                            Defaults to undefined.
      * \param interval (number) - Seconds between checking screen for main menu.
      *                            Defaults to 5.
      * \param timeout (number)  - Seconds between checking screen for main menu.
      *                            Defaults to 30.
      *
      * \return ATVAXElement element on success, undefined on failure
      **/
    mainmenusettings.fromMenuAndHighlight = function fromMenuAndHighlight(name, count, interval, timeout) {
        return(this.fromMenu(interval, timeout).moveToItemWithName(name, count));
    }

    /**
      * Function: mainmenusettings.fromMenuAndSelect(name, count, interval, timeout)
      *     Gets the _appMenu and waits for it to be displayed, checking every
      *     interval seconds until displayed or timeout is reached. Then moves
      *     to and selects to the count'th instance of name.
      *
      *     Throws exception if we cannot get to name.
      *
      * \param name (string)     - The localized name for the menu item to move to.
      * \param count (number)    - When there is more than one item with the same
      *                            name, then a count is required to select past the
      *                            first one. Notice count == undefined (assumes there
      *                            can only be 1) which is different than count == 1.
      *                            Defaults to undefined.
      * \param interval (number) - Seconds between checking screen for main menu.
      *                            Defaults to 5.
      * \param timeout (number)  - Seconds between checking screen for main menu.
      *                            Defaults to 30.
      *
      * \return true on success, throw exception on failure
      **/
    mainmenusettings.fromMenuAndSelect = function fromMenuAndSelect(name, count, interval, timeout) {
        var item = this.fromMenuAndHighlight(name, count, interval, timeout);
        if ( ! item ) {
            throw new Error("Failed to get to item \"" + name + "\"");
        }

        this.appMenu().selectIntoPage();
        return(true);
    }

    /**
      * Function: mainmenusettings.goTo(interval, timeout)
      *     Go to the _appMenu and waits for it to be displayed, checking every
      *     interval seconds until displayed or timeout is reached.
      *
      *     Throws exception if goTo reset function fails.
      *
      * \param interval (number) - Seconds between checking screen for main menu.
      *                            Defaults to 5.
      * \param timeout (number)  - Seconds between checking screen for main menu.
      *                            Defaults to 30.
      *
      * \return ATVAPageList object on success, undefined on failure
      **/
    mainmenusettings.goTo = function goTo(interval, timeout) {
        // Need to reset list when going into
        this.appMenu().resetList();
        if ( ! settings.goToAndSelect(this.localizedString(this.kMainMenuKey))) {
            throw new Error("Failed getting back to \"" + this.localizedString(this.kMainMenuKey) + "\"");
        }
        return(this.fromMenu(interval, timeout));
    }

    /**
      * Function: mainmenusettings.goToAndHighlight(name, count, interval, timeout)
      *     Go to the _appMenu and waits for it to be displayed, checking every
      *     interval seconds until displayed or timeout is reached. Then moves
      *     to and highlight to the count'th instance of name.
      *
      *     Throws exception if goTo reset function fails.
      *
      * \param name (string)     - The localized name for the menu item to move to.
      * \param count (number)    - When there is more than one item with the same
      *                            name, then a count is required to select past the
      *                            first one. Notice count == undefined (assumes there
      *                            can only be 1) which is different than count == 1.
      *                            Defaults to undefined.
      * \param interval (number) - Seconds between checking screen for main menu.
      *                            Defaults to 5.
      * \param timeout (number)  - Seconds between checking screen for main menu.
      *                            Defaults to 30.
      *
      * \return ATVAXElement element on success, undefined on failure
      **/
    mainmenusettings.goToAndHighlight = function goToAndHighlight(name, count, interval, timeout) {
        this.goTo();
        return(this.fromMenuAndHighlight(name, count, interval, timeout));
    }

    /**
      * Function: mainmenusettings.goToAndSelect(name, count, interval, timeout)
      *     Go to the _appMenu and waits for it to be displayed, checking every
      *     interval seconds until displayed or timeout is reached. Then moves
      *     to and selects to the count'th instance of name.
      *
      *     Throws exception if goTo reset function fails.
      *
      *     Throws exception if we cannot get to name.
      *
      * \param name (string)     - The localized name for the menu item to move to.
      * \param count (number)    - When there is more than one item with the same
      *                            name, then a count is required to select past the
      *                            first one. Notice count == undefined (assumes there
      *                            can only be 1) which is different than count == 1.
      *                            Defaults to undefined.
      * \param interval (number) - Seconds between checking screen for main menu.
      *                            Defaults to 5.
      * \param timeout (number)  - Seconds between checking screen for main menu.
      *                            Defaults to 30.
      *
      * \return true on success, throw exception on failure
      **/
    mainmenusettings.goToAndSelect = function goToAndSelect(name, count, interval, timeout) {
        this.goTo();
        return(this.fromMenuAndSelect(name, count, interval, timeout));
    }

    /**
      * Function: mainmenusettings.fromMenuSetItem(name, showState)
      *     Gets the _appMenu and waits for it to be displayed, then set item to Show
      *     or Hide.
      *
      * \param name (string)     - The localized name for the menu item to move to.
      * \param toShow (boolean)  - true  - Sets state to Show
      *                            false - Set state to Hide
      *
      * \return true on success, throw exception on failure
      **/
    mainmenusettings.fromMenuSetItem = function fromMenuSetItem(name, toShow) {
        var theElement = this.fromMenu().elementForName(name);
        if ( ! theElement) {
            throw new Error("Failed to find \"" + name + "\"");
        }
        if (theElement.value() == this.localizedString(this.kParentalControlsRestrictedKey)) {
            throw new Error("Cannot changed value of \"" + name + "\" because it is restricted");
        }
        var valueToSet = undefined;
        if (toShow) {
            if (theElement.value() == this.localizedString(this.kParentalControlsShowKey)) {
                // nothing to do
                return(true);
            }
            valueToSet = this.localizedString(this.kParentalControlsShowKey);
        }
        else {
            if (theElement.value() == this.localizedString(this.kParentalControlsHideKey)) {
                // nothing to do
                return(true);
            }
            valueToSet = this.localizedString(this.kParentalControlsHideKey);
        }

        // need to set
        var focus = this.appMenu().moveToItemWithName(name);
        if ( ! focus) {
            throw new Error("Failed to get to \"" + name + "\"");
        }

        // toggle until set or (toggling too much)
        for (var i = 0; i < 5; ++i) {
            this.appMenu().buttonSelect();

            focus = this.appMenu().itemWithFocus();
            if ( ! focus) {
                throw new Error("Failed to get to focus for \"" + name + "\"");
            }
            if (focus.value() == valueToSet) {
                return(true)
            }
        }

        throw new Error("Failed to set value for \"" + name + "\" to \"" + valueToSet + "\"");
    }

    /**
      * Function: mainmenusettings.goToAndSetItem(name, toShow)
      *     Go to the _appMenu and waits for it to be displayed, the set item to Show or
      *     Hide.
      *
      * \param name (string)     - The localized name for the menu item to move to.
      * \param toShow (boolean)  - true  - Sets state to Show
      *                            false - Set state to Hide
      *
      * \return true on success, throw exception on failure
      **/
    mainmenusettings.goToAndSetItem = function goToAndSelect(name, toShow) {
        this.goTo();
        return(this.fromMenuSetItem(name, toShow));
    }
}
