
/**
 *  Function: netflixTest_browseMovieGenreAtAndPlayFor
 *      Go to Netflix>Genres><category>, go to row, column and play for for time
 *
 *  \param category (string) - category to go to
 *  \param row (string) - row of item to play
 *  \param column (string) - column of item to play
 *  \param secondsToPlay (string) - time to play item
 *   
 *  \return 1 if success, otherwise 0
 **/
function netflixTest_browseMovieGenreAtAndPlayFor() {
    if (this.argv.length < 4) {
        this.testError = "Category, row, column, and seconds to play are required";
        return this.testResult = 0;
    }
    var category = this.argv[0];
    var row = parseInt(this.argv[1]);
    var column = parseInt(this.argv[2]);
    var secondsToPlay = parseInt(this.argv[3]);

    var dp = netflix.goToGenresCategoryMenuSelectRowColumn(category, row, column);
    if ( ! dp) {
        this.testError = "Fail to find movie at row = " + row + " column = " + column;
        return this.testResult = 0;
    }

    if ( ! dp.isDetailsPage()) {
        UIALogger.logMessage("Not a movie at title = \"" + dp.title() + "\"");
        this.testResult = 1;
        return(this.testResult);
    }

    if (secondsToPlay == 0) {
        // Browse only, do not play
    }
    else if (secondsToPlay == -1) {
        // Play until done

        if ( ! dp.playAndWaitForEnd(false)) {
                throw new Error("Failed to playback for movie at title = \"" + dp.title() + "\"");
        }
    }
    else {
        var vp = dp.play(false);
        if ( ! vp) {
            throw new Error("Failed to playback for movie at title = \"" + dp.title() + "\"");
        }

        atvaTarget.delay(secondsToPlay);
        vp.endPlayback();
    }

    this.testResult = 1;
    return(this.testResult);
}

/**
 *  Function: netflixTest_browseMovieGenreAndPlayAllForRowFor
 *      Go to Netflix>Genres><category>, go to row and play all items for time
 *
 *  \param category (string) - category to go to
 *  \param row (string) - row of item to play
 *  \param secondsToPlay (string) - time to play item
 *   
 *  \return 1 if success, otherwise 0
 **/
function netflixTest_browseMovieGenreAndPlayAllForRowFor() {
    if (this.argv.length < 3) {
        this.testError = "Category, row, and seconds to play are required";
        return this.testResult = 0;
    }
    var category = this.argv[0];
    var row = parseInt(this.argv[1]);
    var secondsToPlay = parseInt(this.argv[2]);
    var toPlayCount = 0;
    var column = 0;

    if (this.argv.length > 3) {
        toPlayCount = parseInt(this.argv[3]);
        if (toPlayCount == 0) {
            this.testError = "Play count needs to be more than 0 (>0 for all).";
            return this.testResult = 0;
        }
    }
    else {
        toPlayCount = -1;
    }

    var dp = netflix.goToGenresCategoryMenuSelectRowColumn(category, row, column);
    if ( ! dp) {
        this.testError = "Fail to find movie at row = " + row + " column = " + column;
        return this.testResult = 0;
    }

    while (dp) {
        if ( ! dp.isDetailsPage()) {
            UIALogger.logMessage("Not a movie at title = \"" + dp.title() + "\"");
        }
        else {
            if (secondsToPlay == 0) {
                // Browse only, do not play
            }
            else if (secondsToPlay == -1) {
                // Play until done

                if ( ! dp.playAndWaitForEnd(false)) {
                    this.testError = "Failed to playback for movie at title = \"" + dp.title() + "\"";
                    return this.testResult = 0;
                }
            }
            else {
                var vp = dp.play(false);
                if ( ! vp) {
                    this.testError = "Failed to playback for movie at title = \"" + dp.title() + "\"";
                    return this.testResult = 0;
                }

                atvaTarget.delay(secondsToPlay);
                vp.endPlayback();
            }
        }

        ++column;
        if (column == toPlayCount) {
            dp = undefined;
        }
        else {
            dp = netflix.goToGenresCategoryMenuSelectNext();
        }
    }

    // Successfully played them all
    return this.testResult = 1;
}


/**
 *  Function: netflixTest_browseMovieGenreAndPlayAllForRowFor
 *      Go to Netflix>Genres><category>, go to row and play all items for time
 *
 *  \param category (string) - category to go to
 *  \param row (string) - row of item to play
 *  \param secondsToPlay (string) - time to play item
 *   
 *  \return 1 if success, otherwise 0
 **/
/**
 *       Function: netflixTest_browseDownMovieGenreAndPlayAllFor
 *           Go to Netflix>Genres><category>, start at row, column and play movie for
 *           time. Then select shelfColumn in bookshelf and play. Repeat until howDeep.
 *   
 *  \param category (string) - category to go to
 *  \param row (string) - row of movie to start with
 *  \param column (string) - column of movie to start with
 *  \param shelfColumn (string) - column of movie in self to select
 *  \param howDeep (string) - number of pages deep to go
 *  \param secondsToPlay (string) - time to play item
 *   
 *  \return 1 if found, otherwise 0
 **/

function netflixTest_browseDownMovieGenreAndPlayAllFor() {
    if (this.argv.length < 6) {
        this.testError = "Top movie row, column, shelf column, and how deep need to be given.";
        return this.testResult = 0;
    }
    var category = this.argv[0];
    var row = parseInt(this.argv[1]);
    var column = parseInt(this.argv[2]);
    var shelfColumn = parseInt(this.argv[3]);
    var howDeep = parseInt(this.argv[4]);
    var secondsToPlay = parseInt(this.argv[5]);
    var currentDepth = 0;

    var dp = netflix.goToGenresCategoryMenuSelectRowColumn(category, row, column);
    if ( ! dp) {
        this.testError = "Fail to find movie at row = " + row + " column = " + column;
        return this.testResult = 0;
    }

    while (dp) {
        if ( ! dp.isDetailsPage()) {
            UIALogger.logMessage("Not a movie at title = \"" + dp.title() + "\"");
        }
        else {
            if (secondsToPlay == 0) {
                // Browse only, do not play
            }
            else if (secondsToPlay == -1) {
                // Play until done

                if ( ! dp.playAndWaitForEnd(false)) {
                    this.testError = "Failed to playback for movie at title = \"" + dp.title() + "\"";
                    return this.testResult = 0;
                }
            }
            else {
                var vp = dp.play(false);
                if ( ! vp) {
                    this.testError = "Failed to playback for movie at title = \"" + dp.title() + "\"";
                    return this.testResult = 0;
                }

                atvaTarget.delay(secondsToPlay);
                vp.endPlayback();
            }
        }

        ++currentDepth;
        if (currentDepth > howDeep) {
            dp = undefined;
        }
        else {
            var columnToSelect = shelfColumn;
            var totalColumns = dp.numberOfItemsInBookshelf();
            if (columnToSelect > (totalColumns - 1)) {
                columnToSelect = (totalColumns - 1);
            }

            if (columnToSelect < 0) {
                UIALogger.logMessage("No movies in self, this can be normal");
                return this.testResult = 1;
            }

            var movieItem = dp.moveToBookshelfItem(columnToSelect, true);
            if ( ! movieItem) {
                this.testError = "Fail to find movie in bookself";
                return this.testResult = 0;
            }

            dp = netflix.enterItem(dp, movieItem);
            if ( ! dp) {
                this.testError = "Fail to enter movie in bookself";
                return this.testResult = 0;
            }
        }
    }

    return this.testResult = 1;
}

/**
 *  Function: netflixTest_searchForMovie
 *      Search for movie through Netflix>Search.
 *
 *  \param movieTitle (string) - movie title
 *   
 *  \return 1 if success, otherwise 0
 **/
function netflixTest_searchForMovie() {
    if (this.argv.length < 1) {
        this.testError = "No movie title supplied.";
        return this.testResult = 0;
    }
    var movieTitle = this.argv[0];
    if ( ! netflix.goToSearchMenuSearch(movieTitle)) {
        this.testError = "Fail to search for movie";
        return this.testResult = 0;
    }
    else {
        return this.testResult = 1;
    }
}

/**
 *  Function: netflixTest_searchForMovieAndPlay
 *      Search for movie through Netflix>Search, and play
 *
 *  \param movieTitle (string)       - movie title
 *  \param returnToMainMenu (string) - Return to Main Menu>Settings
 *   
 *  \return 1 if success, otherwise 0
 **/
function netflixTest_searchForMovieAndPlay() {
    if (this.argv.length < 1) {
        this.testError = "No timeout supplied.";
        return this.testResult = 0;
    }
    var movieTitle = this.argv[0];
    var returnToMainMenu = false;
    if (this.argv.length >= 2) {
        returnToMainMenu = true;
    }
    var dp = netflix.goToSearchMenuSearch(movieTitle);
    if ( ! dp) {
        this.testError = "Fail to search for movie";
        this.testResult = 0;
    }
    else {
        if ( ! dp.playAndWaitForEnd(false)) {
            this.testError = "Fail to play for movie";
            this.testResult = 0;
        }
        else {
            this.testResult = 1;
        }
    }

    if (returnToMainMenu) {
        ATVAPageMainMenu.goToAndSelect("Settings");
    }
    return this.testResult;
}

/**
 *  Function: netflixTest_searchForMovieAndPlayFor
 *      Search for movie through Netflix>Search, and play for secondsToPlay
 *
 *  \param movieTitle (string)    - movie title
 *  \param secondsToPlay (number) - seconds to play movie.
 *   
 *  \return 1 if success, otherwise 0
 **/
function netflixTest_searchForMovieAndPlayFor() {
    if (this.argv.length < 2) {
        this.testError = "movie title and time to play must be specified";
        return this.testResult = 0;
    }
    var movieTitle = this.argv[0];
    var secondsToPlay = this.argv[1];
    var dp = netflix.goToSearchMenuSearch(movieTitle);
    if ( ! dp) {
        this.testError = "Fail to search for movie";
        this.testResult = 0;
    }
    else {
        var vp = dp.play(false);
        if ( ! vp) {
            this.testError = "Fail to play for movie";
            this.testResult = 0;
        }
        else {
            atvaTarget.delay(secondsToPlay);
            vp.endPlayback();
            this.testResult = 1;
        }
    }
    return this.testResult;
}

/**
 *  Function: netflixTest_searchForMovieAndPlayAndPauseFor
 *      Search for movie through Netflix>Search, and pause for secondsToPause
 *
 *  \param movieTitle (string)     - movie title
 *  \param secondsToPause (number) - seconds to play movie.
 *   
 *  \return 1 if success, otherwise 0
 **/
function netflixTest_searchForMovieAndPlayAndPauseFor() {
    if (this.argv.length < 2) {
        this.testError = "movie title and time to pause must be specified";
        return this.testResult = 0;
    }
    var movieTitle = this.argv[0];
    var secondsToPause = this.argv[1];
    var dp = netflix.goToSearchMenuSearch(movieTitle);
    if ( ! dp) {
        this.testError = "Fail to search for movie";
        this.testResult = 0;
    }
    else {
        var vp = dp.play(false);
        if ( ! vp) {
            this.testError = "Fail to play for movie";
            this.testResult = 0;
        }
        else {
            vp.pause();
            atvaTarget.delay(secondsToPause);
            vp.dismissScreenSaver();
            vp.unpause();
            vp.endPlayback();
            this.testResult = 1;
        }
    }
    return this.testResult;
}
