/*
	File:		gpus_init.c

	Copyright:	© 1999-2010 by Apple Inc., all rights reserved.

	Writers:
		Bob Beretta
*/

#include <strings.h>

#include <IOKit/IOKitLib.h>

#include "glr_config.h"

#include <GPUSupport/gpus_assert.h>
#include <GPUSupport/gpus_print.h>
#include <GPUSupport/gpus_internal.h>
#include <GPUSupport/gpus_io_data.h>
#include <GPUSupport/gpus_device.h>
#include <GPUSupport/gpus_shared.h>
#include <GPUSupport/gpus_context.h>
#include <GPUSupport/gpus_compute_context.h>
#include <GPUSupport/gpus_framebuffer.h>
#include <GPUSupport/gpus_vertex_array.h>
#include <GPUSupport/gpus_texture.h>
#include <GPUSupport/gpus_sampler.h>
#include <GPUSupport/gpus_buffer.h>
#include <GPUSupport/gpus_program.h>
#include <GPUSupport/gpus_init.h>
#include <GPUSupport/gpus_memory.h>
#include <GPUSupport/gpus_queue.h>
#include <GPUSupport/gpus_compute_program.h>
#include <GPUSupport/gpus_query.h>

// Instead of including all of cl_driver_interface.h.
int glrCompFlush(void* queue);

// Here is the instantiation of gpus_io_data
struct GPUSIODataRec gpus_io_data;

void gldInitializeLibrary(io_service_t *service, uint8_t *fbindex, uint32_t mask,
	gldGetSurfacePropertiesPtr propsFunc, gldGetNewSurfacePtr surfFunc,
	uint32_t options)
{
	GPUSIOData io_data = &gpus_io_data;

	bzero(io_data, sizeof(struct GPUSIODataRec));

	io_data->struct_sizes.device           = sizeof(struct GLDDeviceRec);
	io_data->struct_sizes.shared           = sizeof(struct GLDShareGroupRec);
	io_data->struct_sizes.context          = sizeof(struct GLDContextRec);
	io_data->struct_sizes.compute_context  = sizeof(struct GLDComputeContextRec);
	io_data->struct_sizes.framebuffer      = sizeof(struct GLDFramebufferRec);
	io_data->struct_sizes.vertex_array     = sizeof(struct GLDVertexArrayRec);
	io_data->struct_sizes.texture          = sizeof(struct GLDTextureRec);
	io_data->struct_sizes.sampler          = sizeof(struct GLDSamplerRec);
	io_data->struct_sizes.buffer           = sizeof(struct GLDBufferRec);
	io_data->struct_sizes.program          = sizeof(struct GLDProgramRec);
	io_data->struct_sizes.pipeline_program = sizeof(struct GLDPipelineProgramRec);
	io_data->struct_sizes.memory_plugin    = sizeof(struct GLDMemoryPluginRec);
	io_data->struct_sizes.queue            = sizeof(struct GLDQueueRec);
	io_data->struct_sizes.compute_module   = sizeof(struct GLDComputeModuleRec);
	io_data->struct_sizes.compute_program  = sizeof(struct GLDComputeProgramRec);
	io_data->struct_sizes.kernel           = sizeof(struct GLDKernelRec);
	io_data->struct_sizes.query            = sizeof(struct GLDQueryRec);

	io_data->dispatch.initialize_device_func             = glrInitializeDevice;
	io_data->dispatch.terminate_device_func              = glrTerminateDevice;
	#if !defined(GPUS_MERCURY) || !IOA_USE_EVENTOLATOR
	io_data->dispatch.test_time_stamp_func               = glrTestTimeStamp;
	#endif
	#ifndef GPUS_MERCURY
	io_data->dispatch.get_init_time_stamp_context_func   = glrGetInitTimeStampContext;
	io_data->dispatch.get_init_time_stamp_queue_func     = glrGetInitTimeStampQueue;
	io_data->dispatch.get_init_time_stamp_comp_func      = glrGetInitTimeStampComp;
	#endif
	#if GLR_PIPEPROG_HAS_CTX_BUILD
	io_data->dispatch.release_vend_ctx_pipe_prog_func    = glrReleaseVendCtxPipeProg;
	#else
	io_data->dispatch.release_vend_ctx_pipe_prog_func    = NULL;
	#endif
	io_data->dispatch.comp_flush_func                    = glrCompFlush;
	io_data->dispatch.set_cookie_no_program_func         = gpumSetCookieNoProgramExtern;
	#if GPUS_GST_SUPPORT && GLR_HAS_PERF_COUNTERS
	io_data->dispatch.get_int_dd_func                    = glrGetIntDD;
	#endif

	#if GPUS_SUPPORT_GPU_RESTART_REPORT
	#if GLK_SUPPORT_GPU_RESTART_REPORT
	io_data->config.has_gpu_restart_report               = true;
	#endif
	#else // !GPUS_SUPPORT_GPU_RESTART_REPORT
	#if GLK_SUPPORT_GPU_RESTART_REPORT
	#error
	#endif
	#endif // !GPUS_SUPPORT_GPU_RESTART_REPORT
	#if GLR_CLIENT_ALLOC_QUERY_BUFFER
	io_data->config.client_alloc_query_buffer            = true;
	#endif // GLR_CLIENT_ALLOC_QUERY_BUFFER

	gpusCreateMallocZone(io_data->mallocZone, "GPUSupport_mallocZone");
	gpusAssert(io_data->mallocZone != NULL);
	glrAllocVendorMallocZones(&io_data->vendorMallocZones, &io_data->vendorMallocZoneCount);
	gpusAssert((io_data->vendorMallocZones == NULL) == (io_data->vendorMallocZoneCount == 0));

	// Call into GPUSupport with the IOData for this plugin.
	gpumInitializeIOData(service, fbindex, mask, propsFunc, surfFunc, options,
		io_data, glrPopulateIODataSizes);
}

void gldTerminateLibrary(void)
{
	// Call into GPUSupport with the IOData for this plugin.
	gpumTerminateIOData(&gpus_io_data);
	
	glrFreeVendorMallocZones(gpus_io_data.vendorMallocZones);
	
	/* Deallocate GPUSupport zone */
	gpusDestroyMallocZone(gpus_io_data.mallocZone);
}

bool32_t gldGetVersion(int32_t *major, int32_t *minor, int32_t *build, bitfield32_t *driverID)
{
	if(!gpus_io_data.devices)
	{
		gpusPrintErr("%s: No IO service, returning false", __FUNCTION__);
		return false;
	}
	else
	{
		*major    = GLD_MAJOR_VERSION;
		*minor    = GLD_MINOR_VERSION;
		*build    = GLD_BUILD_VERSION;
		*driverID = glrGetDriverID();

		#if GLR_DEBUG
		gpusAssert(!(*driverID & ~GLD_DRIVER_ID_MASK));
		#else
		gpusAssert(!(*driverID & (~GLD_DRIVER_ID_MASK | GLD_DEBUG_DRIVER_BIT)));
		#endif

		return true;
	}
}

